<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Library of functions and constants for module tlive
 *
 * This file should have two well differenced parts:
 *   - All the core Moodle functions,  neeeded to allow
 *     the module to work integrated in Moodle.
 *   - All the tlive specific functions,  needed
 *     to implement all the module logic. Please,  note
 *     that,  if the module become complex and this lib
 *     grows a lot,  it's HIGHLY recommended to move all
 *     these module specific functions to a new php file,
 *     called "locallib.php" (see forum,  quiz...). This will
 *     help to save some memory when Moodle is performing
 *     actions across all modules.
 *
 * @package    mod_tlive
 * @copyright  2013  Teleskill Italia S.r.l
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 **/

define('TLIVE_INCLUDE_TEST',  1);
define('TLIVE_NEWLINE',  "\r\n");
define('TLIVE_INDENT',  '    ');
/* TABS */
define('TLIVE_NOTAB',  0);
define('TLIVE_TAB1',  1);
define('TLIVE_TAB2',  2);
/* PAGES
 * pages of tab 1
 * no pages foreseen for first tab
 * pages of tab 2 */
define('TLIVE_TAB2_PAGE1',  '1');
define('TLIVE_TAB2_PAGE1NAME',  'tab2page1');
define('TLIVE_TAB2_PAGE2',  '2');
define('TLIVE_TAB2_PAGE2NAME',  'tab2page2');
/*
 * Given an object containing all the necessary data,
 * (defined by the form in mod_form.php) this function
 * will create a new instance and return the id number
 * of the new instance.
 *
 * @param object $tlive An object from the form in mod_form.php
 * @return int The id of the newly inserted tlive record
 */
require_once($CFG->dirroot.'/mod/tlive/weekcalendar/event_availability.php');
require_once($CFG->dirroot.'/calendar/lib.php');

/**
 * This function add a new tlive instance.
 * @param object $tlive
 * @param object $mform
 * @return int new tlive instance id
 */
function tlive_add_instance($tlive, $mform) {
    global $CFG, $DB;

    $checkboxes = array('emailconfman',  'opentoblank');
    foreach ($checkboxes as $checkbox) {
        if (empty($tlive->{$checkbox})) {
            $tlive->{$checkbox} = 0;
        }
    }

    $tlive->timecreated = time();

    /* It rebuilds complete datetime format
     * Start date temporary storage */
    $startdate = $tlive->startdate;
    $tlive->startdate = $tlive->startdate + ($tlive->starthour) * 3600 + ($tlive->startminute) * 60;
    $tlive->enddate =  $startdate + ($tlive->endhour) * 3600 + ($tlive->endminute) * 60;

    /* Check availability places */
    $maxusers = $CFG->tlive_maxusers;

     /* if scheduling is disabled, don't use startdate and enddate*/
    if (!$CFG->tlive_disabledate) {

        $success = event_availability($maxusers, $tlive->users, $tlive->startdate, $tlive->enddate);
        if (!$success) {
            print_error('places_not_available', 'tlive');
        }
    }

    $xmlanswer = tlive_createroom($tlive, $mform);
    if (!tlive_readxmlanswer($xmlanswer,  'errorcode')) {
        $tlive->roomid = tlive_readxmlanswer($xmlanswer,  'roomid');
    }
    if (!isset($tlive->roomid)) {
        print_error('roomcreationfailed', 'tlive');
    }

    if ($tliveid = $DB->insert_record('tlive',  $tlive)) {

        if (!empty($tlive->roomid)) {
            if ($tlive->emailconfman ) {
                tlive_sendemail($tlive,  'add');
            }
        }

        if (!$CFG->tlive_disabledate) {
            tlive_addcalendarevent($tlive, $tliveid);
        }
    }
    return $tliveid;
}

/**
 * Given an object containing all the necessary data,
 * (defined by the form in mod_form.php) this function
 * will update an existing instance with new data.
 *
 * @param object $tlive tlive object instance
 * @param object $mform An object from the form in mod_form.php
 * @return boolean Success/Fail
 */
function tlive_update_instance($tlive, $mform) {
    global $CFG, $DB;

    $checkboxes = array('emailconfman',  'opentoblank');
    foreach ($checkboxes as $checkbox) {
        if (empty($tlive->{$checkbox})) {
            $tlive->{$checkbox} = 0;
        }
    }
    /* if ($tlive->startdate == $tlive->enddate) {
           $tlive->enddate += 3600;
       } */
    $tlive->timemodified = time();
    $tlive->id = $tlive->instance;

    /* It rebuilds complete datetime format
     * Start date temporary storage */

     $startdate = $tlive->startdate;

     /* set startdate and enddate to 0, if scheduling is disabled */
    if($CFG->tlive_disabledate) {

        $startdate = 0;
    }


    $tlive->startdate = $startdate +  ($tlive->starthour)*3600 + ($tlive->startminute)*60;
    $tlive->enddate =  $startdate +  ($tlive->endhour)*3600 + ($tlive->endminute)*60;

    $oldtlive = $DB->get_record('tlive',  array('id' => $tlive->id));

    /* Check availability places */
    $maxusers = $CFG->tlive_maxusers;
    /* if date scheduling is enabled */
    if (!$CFG->tlive_disabledate) {
        $success = event_availability($maxusers, $tlive->users, $tlive->startdate, $tlive->enddate,  $tlive->id);

        if (!$success) {
            print_error('places_not_available', 'tlive');
        }

    }

    if (empty($oldtlive->roomid)) {
        $xmlanswer = tlive_createroom($tlive, $mform);
        if (!tlive_readxmlanswer($xmlanswer,  'errorcode')) {
            $tlive->roomid = tlive_readxmlanswer($xmlanswer,  'roomid');
        }
    } else {
        $tlive->roomid = $oldtlive->roomid;
        $xmlanswer = tlive_modifyroom($tlive, $mform);
        if (tlive_readxmlanswer($xmlanswer,  'errorcode')) {
            print_error('roomcreationfailed', 'tlive');

        }
    }

    if ($tliveid = $DB->update_record('tlive',  $tlive)) {

        if (!$CFG->tlive_disabledate) {

            tlive_updatecalendarevent($tlive);
        }

        if (!empty($tlive->roomid)) {
            /* startdate */
            $oldstartdate = $oldtlive->startdate;
            $newstartdate = $tlive->startdate;

            /*  conference manager id */
            $oldconfman = $oldtlive->confmanid;
            $newconfman = $tlive->confmanid;

            /* email conference manager */
            $oldemailconfman = $oldtlive->emailconfman;
            $newemailconfman = empty($tlive->emailconfman) ? 0 : 1;

            if ($oldconfman != $newconfman) { /* If conference manager is changed */
                if ($oldemailconfman) { /* And if it sent a mail to notify the assigned role */
                    /* Send a new mail to notify the removal from role*/
                    tlive_sendemail($oldtlive,  'retire');
                }

                if ($newemailconfman) {
                    /* Send a e-mail to new conference manager */
                    tlive_sendemail($tlive,  'add');
                }
            } else {
                /* If conference manager is the same,  send an e-mail to notify him */
                if ($newemailconfman) {
                    if ($oldemailconfman) {
                        if ($oldstartdate != $newstartdate) {  /* If startdate is changed */

                            /* if date scheduling is enabled */
                            if (!$CFG->tlive_disabledate) {
                                tlive_sendemail($tlive,  'update');
                            }
                        }
                    } else {
                        tlive_sendemail($tlive,  'add');
                    }
                }
            }
        }
    }
    return $tliveid;
}

/**
 * Given an ID of an instance of this module,
 * this function will permanently delete the instance
 * and any data that depends on it.
 *
 * @param int $id Id of the module instance
 * @return boolean Success/Failure
 */
function tlive_delete_instance($id) {
    global $DB;

    if (! $tlive = $DB->get_record('tlive',  array('id' => $id))) {
        return false;
    }

    $result = true;

    /* Delete any dependent records here
     * room for an instance is supposed to be never more than one */
    $xmlanswer = tlive_deleteroom($tlive);
    /* if ($tlive->emailconfman && ($tlive->startdate > time())) { */
    if ($tlive->emailconfman) {
        /* se il conference manager era stato avvertito,  disdici l'incarico */
        tlive_sendemail($tlive,  'del');
    }

    if (! $DB->delete_records('tlive',  array('id' => $tlive->id))) {
        $result = false;
    }

    return $result;
}


/**
 * Return a small object with summary information about what a
 * user has done with a given particular instance of this module
 * Used for user activity reports.
 *
 * @param object $course Course object
 * @param object $user User object
 * @param object $mod
 * @param object $tlive Tlive instance
 * @return object
 */
function tlive_user_outline($course,  $user,  $mod,  $tlive) {
    return null;
}


/**
 * Print a detailed representation of what a user has done with
 * a given particular instance of this module,  for user activity reports.
 *
 * @param object $course Course object
 * @param object $user User object
 * @param object $mod
 * @param object $tlive
 * @return boolean
 */
function tlive_user_complete($course,  $user,  $mod,  $tlive) {
    return true;
}


/**
 * Given a course and a time,  this module should find recent activity
 * that has occurred in tlive activities and print it out.
 * Return true if there was output,  or false is there was none.
 *
 * @param object $course Course object
 * @param boolean $isteacher True if is a teacher, false otherwise
 * @param boolean $timestart
 * @return boolean
 */
function tlive_print_recent_activity($course,  $isteacher,  $timestart) {
    return false;  /*  True if anything was printed,  otherwise false */
}


/**
 * Function to be run periodically according to the moodle cron
 * This function searches for things that need to be done,  such
 * as sending out mail,  toggling flags etc ...
 *
 * @return boolean
 * @todo Finish documenting this function
 **/
function tlive_cron() {

    global $DB, $CFG;
    require_once($CFG->dirroot.'/mod/tlive/locallib.php');

    /* Update statistics and completions */
    $daylimit = 2;
    $start = (int) $CFG->tlive_cron_stats_start;
    $end = (int) $CFG->tlive_cron_stats_end;

    $now = (int) date('H', time());

    if (tlive_checktime($start, $end, $now)) {

        mtrace('Updating stats...');
        tlive_cron_updatestats($daylimit);
        mtrace('Finished updating stats');
    }

    /* Check and add new recs in Moodle repository */
    $start = (int) $CFG->tlive_cron_recs_start;
    $end = (int) $CFG->tlive_cron_recs_end;

    if (tlive_checktime($start, $end, $now)) {

        mtrace('Check new recs...');
        //tlive_cron_checkrecs();
        mtrace('Finished updating new recs ');
    }

    return true;
}

/**
 * Must return an array of user records (all data) who are participants
 * for a given instance of tlive. Must include every user involved
 * in the instance,  independient of his role (student,  teacher,  admin...)
 * See other modules as example.
 *
 * @param int $tliveid ID of an instance of this module
 * @return mixed boolean/array of students
 */
function tlive_get_participants($tliveid) {
    return false;
}


/**
 * This function returns if a scale is being used by one tlive
 * if it has support for grading and scales. Commented code should be
 * modified if necessary. See forum,  glossary or journal modules
 * as reference.
 *
 * @param int $tliveid ID of an instance of this module
 * @param int $scaleid
 * @return boolean True if the scale is used by the tlive
 */
 
/*function tlive_scale_used($tliveid, $scaleid) {
    
    global $DB;
   
    if ($scaleid and $DB->record_exists('tlive', array('id' => $tliveid, 'grade' => -$scaleid))) {
        return true;
    } else {
        return false;
    }
}*/


/**
 * Checks if scale is being used by any instance of tlive.
 * This function was added in 1.9
 *
 * This is used to find out if scale used anywhere
 * @param int $scaleid
 * @return boolean True if the scale is used by any tlive
 */
 
/*function tlive_scale_used_anywhere($scaleid) {
	
	global $DB;
	
    if ($scaleid and $DB->record_exists('tlive', array('grade' => -$scaleid))) {
        return true;
    } else {
        return false;
    }
}*/


/**
 * Execute post-install custom actions for the module
 * This function was added in 1.9
 *
 * @return boolean true if success,  false on error
 */
function tlive_install() {
    return true;
}


/**
 * List of features supported in Resource module
 * @param string $feature FEATURE_xx constant for requested feature
 * @return mixed True if module supports feature,  false if not,  null if doesn't know
 */
function tlive_supports($feature) {
    switch($feature) {
        case FEATURE_GROUPS:
            return false;
        case FEATURE_GROUPINGS:
            return false;
        case FEATURE_GROUPMEMBERSONLY:
            return true;
        case FEATURE_MOD_INTRO:
            return true;
        case FEATURE_COMPLETION_TRACKS_VIEWS:
            return false;
        case FEATURE_GRADE_HAS_GRADE:
            return false;
        case FEATURE_GRADE_OUTCOMES:
            return false;
        case FEATURE_BACKUP_MOODLE2:
            return true;
        case FEATURE_COMPLETION_HAS_RULES:
            return true;
        case FEATURE_SHOW_DESCRIPTION:
            return true;    
        default:
            return null;
    }
}

/**
 * This function checks if a user
 * achieves the conditions to complete an activity, respect to a selected rule.
 *
 * @param object $course Course
 * @param object $cm Course-module
 * @param int $userid User ID
 * @param bool $type Type of comparison (or/and; can be used as return value if no conditions)
 * @return bool True if completed, false if not, $type if conditions not set.
 */
function tlive_get_completion_state($course, $cm, $userid, $type) {

    global $CFG, $DB;

    /* Get  details */
    if (!($tlive=$DB->get_record('tlive', array('id'=>$cm->instance)))) {
        throw new Exception("Can't find tlive {$cm->instance}");
    }

    /* Default return value */
    $result = $type;

    /* If enabled, checks activity completion  verifica il completamento del modulo in base ai minuti di frequenza */
    if ($tlive->completiontlive) {
        /* Compute total time of user attendance  */
        $duration = $DB->get_field_sql('SELECT COALESCE(SUM(uls.duration), 0) as duration FROM {tlive_lesson_user_sessions} as uls
            INNER JOIN {tlive_user_sessions} as us ON uls.usersessionid=us.usersessionid
            INNER JOIN {tlive_rooms_sessions} as rs ON us.roomsessionid=rs.roomsessionid
            WHERE us.iduser=? AND rs.roomid=? AND us.iduser<>?', array($userid, $tlive->roomid, $tlive->confmanid));

        /* if attendance>=min_attendance, user has completed activity */
        if ($duration && ($duration/60)>=$tlive->completiontlive) {
            $result = true;
            add_to_log($course->id,  "tlive",  "completion",  "view.php?id=$cm->id", $tlive->name, $cm->id, $userid);
        } else {
            $result = false;
        }
    }

    if ($tlive->completiontlivepoll) {
        /* Count total sum of correct answers */
        $query = 'SELECT COALESCE(SUM(pa.correct), 0) as tot_correct FROM {tlive_users_poll_answers} as upa
        INNER JOIN {tlive_user_sessions} as us ON upa.usersessionid=us.usersessionid
        INNER JOIN {tlive_polls_answers} as pa ON upa.answerid=pa.answerid
        INNER JOIN {tlive_rooms_sessions} as rs ON us.roomsessionid=rs.roomsessionid
        WHERE rs.roomid=? AND us.iduser=? AND pa.correct=1 AND us.iduser<>?';
        $totcorrect = $DB->get_field_sql($query, array($tlive->roomid, $userid, $tlive->confmanid));

        /* if totcorrect>=min_correct_answers, user has completed activity */
        if ($totcorrect>=$tlive->completiontlivepoll) {
            $result = true;
            add_to_log($course->id,  "tlive",  "completion",  "view.php?id=$cm->id", $tlive->name, $cm->id, $userid);
        } else {
            $result = false;
        }

    }

    if ($tlive->completiontliveviewsubmit) {

        /* Check if user has participant role and he has logged room at least once */
        $context = get_context_instance(CONTEXT_COURSE, $course->id);

        /* query following moodle role
        *$query = "SELECT * FROM mdl_log as l
                    INNER JOIN mdl_role_assignments as r ON l.userid=r.userid
                    WHERE r.contextid=? AND r.userid=? AND l.`action`='join' AND r.roleid=5"; */

        /* query following tlive role */
        $query = "SELECT * FROM {log}
                   WHERE userid=? AND `action`='join' AND userid <> ?";

        if ($res = $DB->get_records_sql($query, array($userid, $tlive->confmanid))) {
            $result = true;
            add_to_log($course->id,  "tlive",  "completion",  "view.php?id=$cm->id", $tlive->name, $cm->id, $userid);
        } else {
            $result=false;
        }

    }
        return $result;
}

/* original
function tlive_get_completion_state($course, $cm, $userid, $type) {
  global $CFG, $DB;
     Get  details
    if(!($tlive=$DB->get_record('tlive', array('id'=>$cm->instance)))) {
        throw new Exception("Can't find tlive {$cm->instance}");
    }

    /* if($tlive->completiontlivesubmit){


      /* $duration = $DB->get_field_sql('SELECT SUM(uls.duration)as duration FROM mdl_tlive_lesson_user_sessions as uls
          INNER JOIN mdl_tlive_user_sessions as us ON uls.usersessionid=us.usersessionid
          WHERE us.iduser=?', array($userid));

     /* if($duration && ($duration/60)>=$tlive->completiontlive){
          return true;
      } else {
          return false;
      }

    }else {

      return $type;
    }

} */

/**
 * Function that send a request to create a videoconference room, through an XML
 *
 * @param object $tlive Tlive object instance
 * @param object $mform Form object
 * @return string $xml_answer XML answer from Teleskill Live server
 */
function tlive_createroom($tlive, $mform) {

    global $CFG, $COURSE;
    /*  notice: $tlive is passed as params because during
     * calling of add_instance, $tlive not exists as a global var
     */
    $xmlanswer = tlive_isroomavailable($tlive);
    if (tlive_readxmlanswer($xmlanswer,  'errorcode')) {
        /* The room is not available,  return the message */
        return $xmlanswer;
    } else {
        /* The room is available,  try to create it */
        $xmlrequest = teleskill_beginrequest('1');

        $xmlrequest .= TLIVE_INDENT.'<clientcode>'.$CFG->tlive_clientcode.'</clientcode>'.TLIVE_NEWLINE;

        $startdate = '';
        $enddate = '';

        if (!$CFG->tlive_disabledate) {

            $startdate = gmdate("Y-m-d H:i:s",  $tlive->startdate);
            $enddate = gmdate("Y-m-d H:i:s",  $tlive->enddate);
        }

        /* if scheduling isn't disabled, use startdate and enddate */

        $xmlrequest .= TLIVE_INDENT.'<startdate>'.$startdate.'</startdate>'.TLIVE_NEWLINE;
        $xmlrequest .= TLIVE_INDENT.'<enddate>'.$enddate.'</enddate>'.TLIVE_NEWLINE;


        $xmlrequest .= TLIVE_INDENT.'<title>'.$COURSE->shortname.' - '.$tlive->name.'</title>'.TLIVE_NEWLINE;
        /* if ($tlive->intro) { */
        /* At the moment, not include intro description */
        /* $xmlrequest .= TLIVE_INDENT.'<descr>'.$tlive->intro.'</descr>'.TLIVE_NEWLINE; */
        /* } */

        $xmlrequest .= TLIVE_INDENT.'<users>'.$tlive->users.'</users>'.TLIVE_NEWLINE;
        $context = context_module::instance($tlive->coursemodule);

        /* logo image */
        if ($filename = $mform->get_new_filename('logo')) {

            $itemid = file_get_submitted_draft_itemid('logo');

            $file = $mform->save_stored_file('logo', $context->id, 'mod_tlive', 'rooms',
            $itemid, '/', $filename);

            $basepath = $CFG->wwwroot.'/pluginfile.php/'.$context->id.'/mod_tlive/rooms/'.$itemid;
            $xmlrequest .= TLIVE_INDENT.'<logo>'.$basepath.'/'.$filename.'</logo>'.TLIVE_NEWLINE;

        }


        /* banner image */
         if ($filename = $mform->get_new_filename('banner')) {

            $itemid = file_get_submitted_draft_itemid('banner');

            $file = $mform->save_stored_file('banner', $context->id, 'mod_tlive', 'rooms',
                $itemid, '/', $filename);

            $basepath = $CFG->wwwroot.'/pluginfile.php/'.$context->id.'/mod_tlive/rooms/'.$itemid;
            $xmlrequest .= TLIVE_INDENT.'<banner>'.$basepath.'/'.$filename.'</banner>'.TLIVE_NEWLINE;

        }



        $xmlrequest .= '</ews>';

        /* return XML answer string that must be analyzed from receiving */
        $xmlanswer = tlive_getanswerfrom($xmlrequest);
        if (!tlive_readxmlanswer($xmlanswer,  'errorcode')) {
            $tlive->roomid = tlive_readxmlanswer($xmlanswer,  'roomid');
        }

        return $xmlanswer;
    }
}

/**
 * Function that validates XML answer string
 *
 * @param string $xmlanswer Tlive object instance
 * @param string $tag Form object
 * @return string
 */
function tlive_readxmlanswer($xmlanswer,  $tag) {

    global $CFG;

    if (empty($xmlanswer)) {
        if ($CFG->debug < DEBUG_ALL) {
            $a = get_string('debugall',  'admin');
            $a = get_string('riseupdebuglevel',  'tlive',  $a);
        } else {
            $a = '';
        }
        return get_string('emptyxml', 'tlive',  $a);
    }

    $regexp = "|<$tag>(.*)</$tag>|";
    if (preg_match($regexp,  $xmlanswer,  $matches)) {
        return $matches[1]; /* it is a single value */
    }
}

/**
 * Function that sends an e-mail to user
 *
 * @param object $tlive Tlive object instance
 * @param string $action Type of message (add user,retire..)
 */
function tlive_sendemail($tlive, $action) {
    global $CFG,  $USER,  $COURSE,  $DB;

    $a = new stdClass();
    $a->courseurl = $CFG->wwwroot.'/course/view.php?id='.$COURSE->id;
    $a->coursename = $COURSE->fullname;
    $a->conferencename = $tlive->name;
    $a->timestart = userdate($tlive->startdate);

    switch ($CFG->tlive_disabledate) {

        case 0:
            $string = 'emailbody_'.$action;
            break;
        case 1:
            $string = 'emailbody_'.$action.'_nodate';
            break;

    }

    $htmlbody = get_string($string, 'tlive',$a);
    $subject= get_string('emailsubject', 'tlive',  $a);
    $textbody = strip_tags($htmlbody);
    $recipient = $DB->get_record('user',  array('id' => $tlive->confmanid));
    $from = $DB->get_record('user',  array('id' => $USER->id));

    if (!email_to_user($recipient,  $from,  $subject,  $textbody,  $htmlbody)) {
        $a = new stdClass();
        $a->roomcreator = fullname($from);
        $a->conferencemanager = fullname($recipient);
        print_string('emailnotsent', 'tlive',  $a);
    }
}

/**
 * Function that build the XML request to modify room data
 *
 * @param object $tlive Tlive object instance
 * @param string $mform Form object
 * @return string XML answer
 */
function tlive_modifyroom($tlive, $mform) {
    global $CFG, $COURSE;

    $xmlrequest = teleskill_beginrequest('6');

    $xmlrequest .= TLIVE_INDENT.'<clientcode>'.$CFG->tlive_clientcode.'</clientcode>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<roomid>'.$tlive->roomid.'</roomid>'.TLIVE_NEWLINE;
    /* if scheduling isn't disabled, use startdate and enddate */

    $startdate = '';
    $enddate = '';

    if (!$CFG->tlive_disabledate) {

        $startdate = gmdate("Y-m-d H:i:s",  $tlive->startdate);
        $enddate = gmdate("Y-m-d H:i:s",  $tlive->enddate);
    }


    $xmlrequest .= TLIVE_INDENT.'<startdate>'.$startdate.'</startdate>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<enddate>'.$enddate.'</enddate>'.TLIVE_NEWLINE;

    $xmlrequest .= TLIVE_INDENT.'<title>'.$COURSE->shortname.' - '.$tlive->name.'</title>'.TLIVE_NEWLINE;
    /* if ($tlive->intro) {
         * $xmlrequest .= TLIVE_INDENT.'<descr>'.$tlive->intro.'</descr>'.TLIVE_NEWLINE;
     }
     */

    $xmlrequest .= TLIVE_INDENT.'<users>'.$tlive->users.'</users>'.TLIVE_NEWLINE;

    $context = context_module::instance($tlive->coursemodule);

    /* logo image */
    if ($filename = $mform->get_new_filename('logo')) {

        $itemid = file_get_submitted_draft_itemid('logo');

        $file = $mform->save_stored_file('logo', $context->id, 'mod_tlive', 'rooms',
        $itemid, '/', $filename,true);

        $basepath = $CFG->wwwroot.'/pluginfile.php/'.$context->id.'/mod_tlive/rooms/'.$itemid;
        $xmlrequest .= TLIVE_INDENT.'<logo>'.$basepath.'/'.$filename.'</logo>'.TLIVE_NEWLINE;

    }


    /* banner image */
     if ($filename = $mform->get_new_filename('banner')) {

        $itemid = file_get_submitted_draft_itemid('banner');

        $file = $mform->save_stored_file('banner', $context->id, 'mod_tlive', 'rooms',
            $itemid, '/', $filename,true);

        $basepath = $CFG->wwwroot.'/pluginfile.php/'.$context->id.'/mod_tlive/rooms/'.$itemid;
        $xmlrequest .= TLIVE_INDENT.'<banner>'.$basepath.'/'.$filename.'</banner>'.TLIVE_NEWLINE;

    }

    $xmlrequest .= '</ews>';

    return tlive_getanswerfrom($xmlrequest);
}

/**
 * Function that build the XML request to delete room
 *
 * @param object $tlive Tlive object instance
 * @return string XML answer
 */
function tlive_deleteroom($tlive) {
    global $CFG;

    /*  notice: $tlive is passed as params because during
     * calling of delete_instance, $tlive not exists as a global var
     */

    $xmlrequest = teleskill_beginrequest('2');

    $xmlrequest .= TLIVE_INDENT.'<clientcode>'.$CFG->tlive_clientcode.'</clientcode>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<roomid>'.$tlive->roomid.'</roomid>'.TLIVE_NEWLINE;
    $xmlrequest .= '</ews>';

    return tlive_getanswerfrom($xmlrequest);
}

/**
 * Function that check room availability
 *
 * @param object $tlive Tlive object instance
 * @return string XML answer
 */
function tlive_isroomavailable($tlive) {
    global $CFG;

    /*  notice: $tlive is passed as params because during
     * calling of create_instance, $tlive not exists as a global var
     */

    $xmlrequest = teleskill_beginrequest('5');

    $xmlrequest .= TLIVE_INDENT.'<clientcode>'.$CFG->tlive_clientcode.'</clientcode>'.TLIVE_NEWLINE;

    /* if scheduling isn't disabled, use startdate and enddate */
    if(!$CFG->tlive_disabledate) {
        $xmlrequest .= TLIVE_INDENT.'<startdate>'.gmdate("Y-m-d H:i:s",  $tlive->startdate).'</startdate>'.TLIVE_NEWLINE;
        $xmlrequest .= TLIVE_INDENT.'<enddate>'.gmdate("Y-m-d H:i:s",  $tlive->enddate).'</enddate>'.TLIVE_NEWLINE;
    }
    $xmlrequest .= TLIVE_INDENT.'<users>'.$tlive->users.'</users>'.TLIVE_NEWLINE;
    $xmlrequest .= '</ews>';

    return tlive_getanswerfrom($xmlrequest);
}

/**
 * Function that build XML request type heading (add,edit,delete room...)
 *
 * @param int $type Request type
 * @return string $xmlrequest XML request
 */
function teleskill_beginrequest($type) {
    $xmlrequest = '<?xml version="1.0" encoding="UTF-8"?>'.TLIVE_NEWLINE;
    $xmllang = (current_language() == 'it_utf8') ? 'IT' : 'EN';
    $xmlrequest .= '<ews type="'.$type.'" lang="'.$xmllang.'">'.TLIVE_NEWLINE;
    return $xmlrequest;
}

/*function tlive_curl_getanswerfrom($xmlrequest) {
    // this function  uses cURL request
    // notice: cURL has problems with proxy servers

    // per far funzionare il cURL con un proxt Ã¨ necessario aggiungere le seguenti tre opzioni:
    // curl_setopt($ch,  CURLOPT_PROXY,  â€œhttp://000.000.000.000:0000â€³);
    // curl_setopt($ch,  CURLOPT_PROXYPORT,  0000);
    // curl_setopt ($ch,  CURLOPT_PROXYUSERPWD,  â€œxxx:xxxâ€�);

    global $CFG;

    $ch = curl_init();
    //echo '$CFG->tlive_serverhost = '.$CFG->tlive_serverhost.'<br />';
    $url = $CFG->tlive_serverhost;
    if ($CFG->tlive_serverport != '80')
        $url .= ':'.$CFG->tlive_serverport;

    curl_setopt($ch,  CURLOPT_URL,  $url);
    curl_setopt($ch,  CURLOPT_RETURNTRANSFER,  1);
    curl_setopt($ch,  CURLOPT_POST,  1);
    curl_setopt($ch,  CURLOPT_TIMEOUT,  10);
    //curl_setopt($ch,  CURLOPT_HEADER,  1); // never uncomment me

    curl_setopt($ch,  CURLOPT_POSTFIELDS,  'message='.urlencode($xmlrequest));

    $xmlanswer = curl_exec($ch);

    if (($CFG->debug >= DEBUG_ALL) && $CFG->debugdisplay) {
        if (!$xmlanswer) {
            $output = 'No cURL data returned for '.$url.'<br />';
            if (curl_error($ch))
                $output .= curl_error($ch).'<br />';
            $info = curl_getinfo($ch);
            if ($info['http_code'] != 200)
                $output .= 'Last received HTTP code: '.$info['http_code'].'<br />';
            echo $output;
        }
    }

    curl_close($ch);

    if (($CFG->debug >= DEBUG_ALL) && $CFG->debugdisplay) {
        echo '$xmlrequest = <textarea cols="90" rows="10">'.$xmlrequest.'</textarea><br />';

        echo '$xmlanswer = <textarea cols="90" rows="10">'.$xmlanswer.'</textarea><br />';
        print_string('reducedebuglevel', 'tlive');
    }
    return $xmlanswer;
}*/

/**
 * Function that sends http request to Teleskill Live server, using fsock
 *
 * @param int $xmlrequest Request type
 * @return string $xmlanswer XML answer
 */
function tlive_getanswerfrom($xmlrequest) {
    /* custom function file_get_contents() that sends data through POST request */
    global $CFG;
    $xmlanswer = false;

    $url = parse_url($CFG->tlive_serverhost);
    $url['protocol']=$url['scheme'].'://';

    $postdata = 'message='.urlencode($xmlrequest);

    $postrequest = 'POST '.$CFG->tlive_serverhost.' HTTP/1.0'.TLIVE_NEWLINE
        .'Host: '.$url['host'].TLIVE_NEWLINE
        .'User-Agent: PHP Script'.TLIVE_NEWLINE
        .'Content-type: application/x-www-form-urlencoded'.TLIVE_NEWLINE
        .'Content-length: '.strlen($postdata).TLIVE_NEWLINE
        .'Connection: close'.TLIVE_NEWLINE.TLIVE_NEWLINE
        .$postdata.TLIVE_NEWLINE.TLIVE_NEWLINE;

    $socket = fsockopen($url['host'],  $CFG->tlive_serverport,  $errno,  $errstr,  30);

    if ($socket) {
        $xmlanswer = '';
        socket_set_timeout($socket,  10);
        fputs($socket,  $postrequest);

        /* discard header */
        $head = fgets($socket);
        if (substr_count($head,  "200 OK") > 0) {
            $headerrow = 0;
            while (!(fgets($socket) == TLIVE_NEWLINE) && $headerrow < 100) {
                ++$headerrow;
            }
            if ($headerrow == 100) {
                return false;
            }
        } else {
            return false;
        }

        while (!feof($socket)) {
            $xmlanswer .= fgets($socket,  4096);
        }
        fclose($socket);
        if (($CFG->debug >= DEBUG_ALL) && $CFG->debugdisplay) {
            echo '$xmlrequest = <textarea cols="90" rows="10">'.$xmlrequest.'</textarea><br />';

            echo '$xmlanswer = <textarea cols="90" rows="10">'.$xmlanswer.'</textarea><br />';
            print_string('reducedebuglevel', 'tlive');
        }
    }

    return $xmlanswer;
}

/**
 * Function used by repository plugin to send a file requested by context and file area
 *
 * @param object $course course instance
 * @param object $cm course module instance
 * @param object $context instance context
 * @param string $filearea file area
 * @param array $args arguments
 * @param boolean $forcedownload force download file
 * @param array $options options
 */
function tlive_pluginfile($course, $cm, $context, $filearea, $args, $forcedownload, array $options=array()) {

    $filepath='/';
    $itemid = array_shift($args);
    $filename = array_pop($args);
    $fs = get_file_storage();
    $file = $fs->get_file($context->id, 'mod_tlive', $filearea, $itemid, $filepath, $filename);
    if (!$file) {
        return false; // The file does not exist.
    }

    send_stored_file($file, 86400, 0, $forcedownload, $options);

}

/**
 * This function add new calendar event for Tlive instance
 *
 * @param object $tlive tlive instance
 * @param object $tliveid tlive id new instance
 */
function tlive_addcalendarevent($tlive, $tliveid) {

    $event = new stdClass();
    $event->name = $tlive->name;
    $event->description = $tlive->intro;
    $event->courseid = $tlive->course;
    $event->groupid      = 0;
    $event->userid       = 0;
    $event->modulename   = 'tlive';
    $event->eventtype    = 'course';
    $event->instance     = $tliveid;
    $event->timestart    = $tlive->startdate;
    $event->timeduration = $tlive->enddate - $tlive->startdate;
    $event->visible      = instance_is_visible('tlive', $tlive);

    calendar_event::create($event);

}

/**
 * This function update calendar event for current Tlive instance
 *
 * @param object $tlive tlive instance
 */
function tlive_updatecalendarevent($tlive) {

    global $DB;

    if ($eventrecord = $DB->get_record('event', array('modulename'=>'tlive', 'instance'=>$tlive->id)) ) {

        $event = calendar_event::load($eventrecord->id);
        $updevent = new stdClass();
        $updevent->name = $tlive->name;
        $updevent->description = $tlive->intro;
        $updevent->timestart    = $tlive->startdate;
        $updevent->timeduration = $tlive->enddate - $tlive->startdate;

        $event->update($updevent);

    }

}
