<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Local functions file of tlive module
 *
 * @package    mod_tlive
 * @copyright  2013  Teleskill Italia S.r.l
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 **/

require_once("$CFG->dirroot/mod/tlive/lib.php");

/**
 * This function checks "doanything" user capability
 * @return bolean
 */
function tlive_user_candoanything() {
    $context = get_context_instance(CONTEXT_SYSTEM);
    return (has_capability('moodle/site:doanything',  $context));
}

/**
 * This function checks if user has capability to be assigned as conference manager
 * @param object $cm Course module
 * @return boolean
 */
function tlive_useris_conferencemanagercandidate($cm) {
    $context = context_module::instance($cm->id);
    return (has_capability('mod/tlive:conferencemanagercandidate',  $context));
}

/**
 * This function checks if user has capability to manage tlive logs
 * @param object $cm Course module
 * @return boolean
 */
function tlive_usercan_managelogs($cm) {
    $context = context_module::instance( $cm->id);
    return (has_capability('mod/tlive:managelogs',  $context));
}

/**
 * This function builds XML request to enable user access
 *
 * @param  object $tlive Tlive instance object
 * @return string  XML answer returned by Teleskill Live
 */
function tlive_enableuser($tlive) {
    global $CFG,  $USER;
    $role = ($tlive->confmanid == $USER->id) ? '2' : '1';
    $lang = strtoupper($USER->lang);
    $xmlrequest = teleskill_beginrequest('3');

    $xmlrequest .= TLIVE_INDENT.'<clientcode>'.$CFG->tlive_clientcode.'</clientcode>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<roomid>'.$tlive->roomid.'</roomid>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<lmsuserid>'.$USER->id.'</lmsuserid>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<role>'.$role.'</role>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<name>'.fullname($USER).'</name>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<lang>'.$lang.'</lang>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<email>'.$USER->email.'</email>'.TLIVE_NEWLINE;
    $xmlrequest .= '</ews>';

    return tlive_getanswerfrom($xmlrequest);
}

/**
 * This function builds XML request to get room's logs
 *
 * @param  int $complete If 1 ask complete log, 0 otherwise
 * @param  object $tlive Tlive instance object
 * @return string  XML answer returned by Teleskill Live
 */
function tlive_askforlogs($complete,  $tlive) {
    global $CFG;

    $xmlrequest = teleskill_beginrequest('16');

    $xmlrequest .= TLIVE_INDENT.'<clientcode>'.$CFG->tlive_clientcode.'</clientcode>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<roomid>'.$tlive->roomid.'</roomid>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<complete>'.$complete.'</complete>'.TLIVE_NEWLINE;
    $xmlrequest .= '</ews>';

    return tlive_getanswerfrom($xmlrequest);
}

/**
 * This function builds XML request to get channel data
 *
 * @return string  XML answer returned by Teleskill Live
 */
function tlive_channeldatarequest() {
    global $CFG;

    $xmlrequest = teleskill_beginrequest('11');

    $xmlrequest .= TLIVE_INDENT.'<clientcode>'.$CFG->tlive_clientcode.'</clientcode>'.TLIVE_NEWLINE;
    $xmlrequest .= '</ews>';

    return tlive_getanswerfrom($xmlrequest);
}

/**
 * This function builds XML request to get recorded room sessions
 *
 * @param int $roomid Room id
 * @param int $fromdate Start date in unix timestamp
 * @param int $todate End date in unix timestamp
 * @return string  XML answer returned by Teleskill Live
 */
function tlive_sessions($roomid, $fromdate, $todate) {
    global $CFG;

    if (empty($roomid)) {
        $xmlrequest = teleskill_beginrequest('21');
    } else {
        $xmlrequest = teleskill_beginrequest('20');
    }

    $xmlrequest .= TLIVE_INDENT.'<clientcode>'.$CFG->tlive_clientcode.'</clientcode>'.TLIVE_NEWLINE;
    if (!empty($roomid)) {
        $xmlrequest .= TLIVE_INDENT.'<roomid>'.$roomid.'</roomid>'.TLIVE_NEWLINE;
    }
    $xmlrequest .= TLIVE_INDENT.'<fromdate>'.$fromdate.'</fromdate>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<todate>'.$todate.'</todate>'.TLIVE_NEWLINE;
    $xmlrequest .= '</ews>';

    return tlive_getanswerfrom($xmlrequest);
}

/**
 * This function saves logs data to db from XML log answer
 *
 * @param string $xmlanswer XML answer returned by Teleskill Live
 */
function tlive_savelogs($xmlanswer) {

    global $DB;

    $xmlstructure = simplexml_load_string($xmlanswer);
    /* Statistics are passed by Tlive in Europe/Rome local time*/
    date_default_timezone_set('Europe/Rome');

    if ($xmlstructure->Room) {

        $room = $xmlstructure->Room;
        $roomid = (int) $xmlstructure->Room->attributes()->ID;

        if (!$DB->get_record('tlive_rooms', array('roomid' => $roomid))) {

            $query="INSERT INTO {tlive_rooms} VALUES (?, 0)";
            $DB->execute($query, array($roomid));
        }

        foreach ($room->RoomSession as $roomsession) {

            $roomsessionid = (int)$roomsession->attributes()->ID;
            $startdate = strtotime($roomsession->attributes()->StartDate);
            $enddate = strtotime($roomsession->attributes()->EndDate);

            if (!$DB->get_record('tlive_rooms_sessions', array('roomsessionid' => $roomsessionid))) {
                $query="INSERT INTO {tlive_rooms_sessions} VALUES (?, ?, ?, ?, ?)";

                /* Notice: you can't use insert_record because primary key (roomid) is already defined,
                 * use execute SQL */
                $DB->execute($query, array($roomsessionid, $roomid, $startdate, $enddate, 0));
            }

            /* if polls sessions exist, save them */
            if ($roomsession->Polls) {
                tlive_getpolls($roomsession->Polls, $roomsessionid);
            }

            /* if users sessions exist, save them */
            if ($roomsession->UserSessions) {
                tlive_getusersessions($roomsession->UserSessions, $roomsessionid);
            }

            /* if lessions sessions exist, save them */
            if ($roomsession->LessonSessions) {
                tlive_getlessonsessions($roomsession->LessonSessions, $roomsessionid);
            }

        }

    }

}

/**
 * This function saves polls data to db, from polls simplexml object
 *
 * @param object $polls Polls data simplexml object
 * @param int $roomsessionid Room session id
 */
function tlive_getpolls($polls, $roomsessionid) {

    global $DB;

    foreach ($polls->Poll as $poll) {

        $pollid = (int)$poll->attributes()->ID;
        /* replace "?" and prevent any quotes */
        //$question = addslashes(str_replace("?",  '',  $poll->attributes()->Question));
        $question = (string)$poll->attributes()->Question;

        if (!$DB->get_record('tlive_polls', array('pollid' => $pollid))) {
            $query="INSERT INTO {tlive_polls} VALUES (?, ?, ?)";
            $DB->execute($query, array($pollid, $question, $roomsessionid));
        }

        if ($poll->Answers) {
            tlive_getanswers($poll->Answers, $pollid);
        }

        if ($poll->PollSessions) {
            tlive_getpollsessions($poll->PollSessions, $pollid);
        }

    }

}

/**
 * This function saves poll answers data to db, from answers simplexml object
 *
 * @param object $answers Answers simplexml object
 * @param int $pollid Poll id
 */
function tlive_getanswers($answers, $pollid) {

    global $DB;

    foreach ($answers->Answer as $answer) {

        $answerid = (int)$answer->attributes()->ID;
        $sequence = (int)$answer->attributes()->Sequence;

        //$answerstr = addslashes(str_replace("?",  '', $answer->attributes()->Answer));
        $answerstr  = (string)$answer->attributes()->Answer;       
        
        $correct = (int)$answer->attributes()->Correct;
        

        if (!$DB->get_record('tlive_polls_answers', array('answerid' => $answerid))) {
            $query="INSERT INTO {tlive_polls_answers} VALUES (?, ?, ?, ?, ?)";
            $DB->execute($query, array($answerid, $pollid, $answerstr, $sequence, $correct));
        }

    }

}

/**
 * This function saves poll sessions data to db, from answers simplexml object
 *
 * @param object $pollsessions Answers simplexml object
 * @param int $pollid Poll id
 */
function tlive_getpollsessions($pollsessions, $pollid) {

    global $DB;

    foreach ($pollsessions->PollSession as $pollsession) {

        $pollsessionid = (int)$pollsession->attributes()->ID;
        $opendate = (int)strtotime($pollsession->attributes()->OpenDate);
        $closedate = (int)strtotime($pollsession->attributes()->CloseDate);        
        

        if (!$DB->get_record('tlive_polls_sessions', array('pollsessionid' => $pollsessionid))) {
            $query="INSERT INTO {tlive_polls_sessions} VALUES (?, ?, ?, ?)";
            $DB->execute($query, array($pollsessionid, $pollid, $opendate, $closedate));
        }

        $useranswers = $pollsession->UserAnswer;
        tlive_getuseranswers($useranswers, $pollsessionid);
    }
}

/**
 * This function saves user answers data to db, from user answers simplexml object
 *
 * @param object $useranswers Answers simplexml object
 * @param int $pollsessionid Poll session id
 */
function tlive_getuseranswers($useranswers, $pollsessionid) {

    global $DB;

    foreach ($useranswers as $useranswer) {

        $useranswerid = (int)$useranswer->attributes()->ID;
        $usersessionid = (int)$useranswer->attributes()->UserSessionID;
        $answerid = (int)$useranswer->attributes()->AnswerID;
        $creationdate = strtotime($useranswer->attributes()->CreationDate);

        if (!$DB->get_record('tlive_users_poll_answers', array('userspollanswerid' => $useranswerid))) {
            $query="INSERT INTO {tlive_users_poll_answers}
                    VALUES (?, ?, ?, ?, ?)";
            $DB->execute($query, array($useranswerid, $pollsessionid, $usersessionid, $answerid, $creationdate));
        }

    }
}

/**
 * This function saves user sessions data to db, from user sessions simplexml object
 *
 * @param object $usersessions Answers simplexml object
 * @param int $roomsessionid Room session id
 */
function tlive_getusersessions($usersessions, $roomsessionid) {

    global $DB;

    foreach ($usersessions->UserSession as $usersession) {

        $usersessionid = (int) $usersession->attributes()->ID;
        $userid = (int)$usersession->attributes()->UserID;
        $roleid = (int)$usersession->attributes()->RoleID;
        $startdate = strtotime($usersession->attributes()->StartDate);
        $enddate = strtotime($usersession->attributes()->EndDate);
        /* non presente nel xml
        * $ipaddress = $usersession->attributes()->IpAddress; */

        if (!$DB->get_record('tlive_user_sessions', array('usersessionid' => $usersessionid))) {
            $query="INSERT INTO {tlive_user_sessions}
                    VALUES (?, ?, ?, ?, ?, ?, ?)";
            $DB->execute($query, array($usersessionid, $userid, $roomsessionid, $roleid, $startdate, $enddate, ''));
        }
    }
}

/**
 * This function saves lesson sessions data to db, from lesson sessions simplexml object
 *
 * @param object $lessonsessions Lesson sessions simplexml object
 * @param int $roomsessionid Room session id
 */
function tlive_getlessonsessions($lessonsessions, $roomsessionid) {

    global $DB;

    foreach ($lessonsessions->LessonSession as $lessonsession) {

        $lessonsessionid = (int) $lessonsession->attributes()->ID;
        $lessonstartdate = strtotime($lessonsession->attributes()->StartDate);
        $lessonenddate = strtotime($lessonsession->attributes()->EndDate);
        $lessonduration = (int)$lessonsession->attributes()->Duration;

        if (!$DB->get_record('tlive_lesson_sessions', array('lessonsessionid' => $lessonsessionid))) {
            $query="INSERT INTO {tlive_lesson_sessions}
                    VALUES (?, ?, ?, ?, ?)";
            $DB->execute($query, array($lessonsessionid, $roomsessionid, $lessonstartdate, $lessonenddate, $lessonduration));
        }
        tlive_setuserlessonsession($lessonsession, $roomsessionid);
    }
}

/**
 * This function compute the effective user lesson session, crossing user sessions data with lesson sessions data
 * and save it to db
 *
 * @param object $lessonsession Lesson session simplexml object
 * @param int $roomsessionid Room session id
 */
function tlive_setuserlessonsession($lessonsession, $roomsessionid) {

    global $DB;

    $lessonstartdate = strtotime($lessonsession->attributes()->StartDate);
    $lessonenddate   = strtotime($lessonsession->attributes()->EndDate);
    $lessonsessionid = (int)$lessonsession->attributes()->ID;

    /* extract all users sessions */
    if ($usersessions = $DB->get_records('tlive_user_sessions', array('roomsessionid' => $roomsessionid))) {

        /* for each user session check if it intersects lesson session, if true, save it to db */
        foreach ($usersessions as $usersession) {

            $cond1 = ($lessonstartdate>= $usersession->startdate && $lessonstartdate<=$usersession->enddate);
            $cond2 = ($lessonenddate>= $usersession->startdate && $lessonenddate<=$usersession->enddate);
            $cond3 = ($lessonstartdate<= $usersession->startdate && $lessonenddate>=$usersession->startdate);

            if ( $cond1 || $cond2 || $cond3 ) {

                /* save user lesson session  */
                $userlessonsessionstart = max($lessonstartdate, $usersession->startdate);
                $userlessonsessionend = min($lessonenddate, $usersession->enddate);
                $usersessionid = $usersession->usersessionid;
                $duration = $userlessonsessionend - $userlessonsessionstart;

                $check = $DB->get_record('tlive_lesson_user_sessions',
                                          array('lessonsessionid' => $lessonsessionid, 'usersessionid' => $usersessionid));
                if (!$check) {
                    $query ="INSERT INTO {tlive_lesson_user_sessions}
                             VALUES (NULL, ?, ?, ?, ?, ?)";
                    $DB->execute($query, array($lessonsessionid, $usersessionid,
                                     $userlessonsessionstart, $userlessonsessionend, $duration));
                }
            }
        }
    }
}

/**
 * This function print an html table that summarizes poll data
 *
 * @param int $pollsessionid Poll session id
 * @return object $table Html object table
 */
function tlive_buildtablepolls($pollsessionid) {

    global $DB;

    $strfdate = get_string('displaydateformat',  'tlive');
    $strftime = get_string('displaytimeformat',  'tlive');
    $query = "SELECT p.question, ps.opendate, ps.closedate
               FROM {tlive_polls_sessions} as ps
               INNER JOIN {tlive_polls} as p ON ps.pollid=p.pollid
               WHERE ps.pollsessionid='$pollsessionid'";
    $poll = $DB->get_record_sql($query);
    $title = get_string('question', 'tlive').": $poll->question,  ".
             get_string('poll_session', 'tlive').' '.userdate($poll->opendate, $strfdate.' '.$strftime).
             " - ".userdate($poll->closedate, $strftime);
    $tlive_roles = array(
    	1 => get_string('participant', 'tlive'),
    	2 => get_string('conferencemanager', 'tlive'),
    	3 => get_string('speaker', 'tlive')
    );         

    /* sort fields */
    $sortparams = tlive_sort_tablebyfield();
    $qrstr = $sortparams->qrStr;
    $sortfield = $sortparams->sort_field;
    $sortdirection = $sortparams->sort_direction;
    $newsortdirection = $sortparams->new_sort_direction;

    /* table heading */
    $headinguser = "<a href='$qrstr&sort=firstname&order=$newsortdirection'>".get_string('user', 'tlive')."</a>";
    $headingrole = "<a href='$qrstr&sort=roleid&order=$newsortdirection'>".get_string('role', 'tlive')."</a>";
    $headinganswer = "<a href='$qrstr&sort=answer&order=$newsortdirection'>".get_string('answer', 'tlive')."</a>";
    $headingcorrect = "<a href='$qrstr&sort=correct&order=$newsortdirection'>".get_string('correct', 'tlive')."</a>";

    $table = new html_table();
    $table->head  = array($headinguser, $headingrole, $headinganswer, $headingcorrect);
    $table->data = array();
    /* title is a custom propriety, not present in html_table class */
    $table->title = $title;
    $table->tablealign = 'center';
    $table->width = '60%';

    $query = "SELECT upa.userspollanswerid, u.firstname, u.lastname, ps.correct, us.roleid,
              ps.answer, us.startdate, us.enddate
              FROM ({tlive_users_poll_answers} AS upa
                INNER JOIN {tlive_user_sessions} AS us ON upa.usersessionid=us.usersessionid
                INNER JOIN {tlive_polls_answers} AS ps ON upa.answerid=ps.answerid)
                INNER JOIN {user} as u ON us.iduser = u.id
                WHERE pollsessionid='$pollsessionid' ORDER BY $sortfield $sortdirection";

    if ($result = $DB->get_records_sql($query)) {

        foreach ($result as $r) {

            if ($r->correct==1) {
                $correct = get_string('answer_correct', 'tlive');
            } else {
                $correct = get_string('answer_wrong', 'tlive');
            }

            $roletext = $tlive_roles[$r->roleid];
            $row = array($r->firstname.' '.$r->lastname, $roletext, $r->answer, $correct);
            array_push($table->data, $row);

        }
    }

    return $table;

}

/**
 * This function print an html table that summarizes room access data
 *
 * @param int $roomsessionid Room session id
 * @param int $lessonsessionid Lesson session id
 * @return object $table Html object table
 */
function tlive_buildtableaccess($roomsessionid, $lessonsessionid=null) {

    global $DB;

    $rsid = optional_param('rsid', 0, PARAM_INT);
    $lsid = optional_param('lsid', 0, PARAM_INT);

    $strfdate = get_string('displaydateformat',  'tlive');
    $strftime = get_string('displaytimeformat',  'tlive');
    
    $tlive_roles = array(
    	1 => get_string('participant', 'tlive'),
    	2 => get_string('conferencemanager', 'tlive'),
    	3 => get_string('speaker', 'tlive')
    );   

    /* Print title */
    if ($rsid != 0 && $lsid != 0) {

        $lessonsessionrecord = $DB->get_record('tlive_lesson_sessions', array('lessonsessionid' => $lsid));

        $title = get_string('access_list', 'tlive')." - ".
        get_string("room_session",  "tlive").userdate($lessonsessionrecord->startdate, $strfdate).' '.get_string("from", "tlive").
        userdate($lessonsessionrecord->startdate, $strftime).' '.get_string("to", "tlive").
        userdate($lessonsessionrecord->enddate, $strftime);

    } else if ($rsid != 0) {

        $roomsessionrecord = $DB->get_record('tlive_rooms_sessions', array('roomsessionid' => $rsid));
        $title = get_string('access_list', 'tlive')." - ".
        get_string("room_session",  "tlive").userdate($roomsessionrecord->startdate, $strfdate).' '.get_string("from", "tlive").
        userdate($roomsessionrecord->startdate, $strftime).' '.
        get_string("to", "tlive").userdate($roomsessionrecord->enddate, $strftime);

    } else {
        $title="";
    }

    /* Sort fields */
    $sortparams = tlive_sort_tablebyfield();
    $qrstr = $sortparams->qrStr;
    $sortfield = $sortparams->sort_field;
    $sortdirection = $sortparams->sort_direction;
    $newsortdirection = $sortparams->new_sort_direction;

    /* table heading */
    $headinguser = "<a href='$qrstr&sort=firstname&order=$newsortdirection'>".get_string('user', 'tlive')."</a>";
    $headingrole = "<a href='$qrstr&sort=roleid&order=$newsortdirection'>".get_string('role', 'tlive')."</a>";
    $headingstart = "<a href='$qrstr&sort=startdate&order=$newsortdirection'>".get_string('start', 'tlive')."</a>";
    $headingend = "<a href='$qrstr&sort=enddate&order=$newsortdirection'>".get_string('end', 'tlive')."</a>";
    /* headingip sort=ipaddress */

    $tableuserssessions = new html_table();
    $tableuserssessions->head  = array($headinguser, $headingrole, $headingstart, $headingend);
    $tableuserssessions->data = array();
    $tableuserssessions->title = $title;
    $tableuserssessions->tablealign = 'center';
    $tableuserssessions->width = '60%';

    /* Flag to call print user lession session task */
    if (isset($lessonsessionid)) {

        $query = "SELECT uls.sessionid, u.firstname, u.lastname, uls.startdate, uls.enddate, us.roleid, us.ipaddress
                   FROM {tlive_lesson_user_sessions} AS uls
                   INNER JOIN {tlive_user_sessions} AS us ON uls.usersessionid=us.usersessionid
                   INNER JOIN {user} AS u ON us.iduser=u.id
                   WHERE uls.lessonsessionid='$lessonsessionid' ORDER BY $sortfield $sortdirection";

    } else {
        $query= "SELECT * FROM {tlive_user_sessions} AS us
                 INNER JOIN {user} AS u ON us.iduser=u.id
                 WHERE us.roomsessionid='$roomsessionid' ORDER BY $sortfield $sortdirection";
    }

    if (!$usersessions = $DB->get_records_sql($query)) {
        echo get_string('nousersessionavailable', 'tlive');
    } else {

        foreach ($usersessions as $u) {            

            $usertext = $u->firstname.' '.$u->lastname;
            $roletext = $tlive_roles[$u->roleid];
            $start = userdate($u->startdate, $strfdate.' '.$strftime);
            $end = userdate($u->enddate, $strfdate.' '.$strftime);
            /* ip = u->ipaddress;
            aux = arr(usertext, roletext, start, end, ip); */
            $aux = array($usertext, $roletext, $start, $end);
            array_push($tableuserssessions->data, $aux);

        }

    }
    return $tableuserssessions;
}

/**
 * This function update room logs and activity completion for each user
 *
 * @param object $tlive Tlive instance object
 * @param object $course Course instance
 * @param object $cm object Course module instance
 */
function tlive_updatestats($tlive, $course, $cm) {

    global $CFG, $DB;
    /* Required to fix bug -> Fatal error: Class 'completion_info' not found  */
    require_once("$CFG->libdir/completionlib.php");

    add_to_log($course->id,  "tlive",  "upd statistics",  "view.php?id=$cm->id&tab=2", $tlive->name, $cm->id);
    $complete = 0;
    $xmlanswer = tlive_askforlogs($complete,  $tlive);
    tlive_savelogs($xmlanswer);
    /* Updating activity completion for all participants */
    $completion = new completion_info($course);

    if ($completion->is_enabled($cm) && ($tlive->completiontlive || $tlive->completiontlivepoll) ) {

        /*  query to extract users from lessions
            $query = 'SELECT DISTINCT us.iduser, rs.roomid, ra.roleid FROM mdl_tlive_lesson_user_sessions as uls
            INNER JOIN mdl_tlive_user_sessions as us ON uls.usersessionid=us.usersessionid
            INNER JOIN mdl_tlive_rooms_sessions as rs ON us.roomsessionid=rs.roomsessionid
            INNER JOIN mdl_role_assignments as ra ON us.iduser=ra.userid
            WHERE rs.roomid=? AND ra.roleid=5';
        */
        /*  query to extract users from room sessions -->if and only if user is a student (roleid=5),
            check activity as completed
            $query = 'SELECT DISTINCT us.iduser FROM mdl_tlive_user_sessions as us
            INNER JOIN mdl_tlive_rooms_sessions as r ON us.roomsessionid = r.roomsessionid
            INNER JOIN mdl_role_assignments as ra ON us.iduser=ra.userid
            WHERE r.roomid=? AND ra.contextid=? AND ra.roleid=5';
        */
        /* query to extract users relative to Teleskill Live role
            -> if and only if user is not a conference managae */
            $query = 'SELECT DISTINCT us.iduser FROM {tlive_user_sessions} as us
            INNER JOIN {tlive_rooms_sessions} as r ON us.roomsessionid = r.roomsessionid
            WHERE r.roomid=?';

        if ($users = $DB->get_records_sql($query, array($tlive->roomid, $tlive->confmanid))) {
            foreach ($users as $user) {

                /* $user_role = get_user_roles($context, $user->iduser);
                updating activity completion for each user */
                $completion->update_state($cm, COMPLETION_COMPLETE, $user->iduser);

            }
        }
    }
}

/**
 * This function exports an input html table to Excel worksheet, using Moodle Excel library
 *
 * @param object $table Html table object
 */
function tlive_exportxls($table) {

    global $CFG;

    /* Moodle Excel lib */
    require_once("$CFG->libdir/excellib.class.php");

    $filename = 'export.xls';
    /* Creating a workbook */
    $workbook = new MoodleExcelWorkbook("-");
    $workbook->send($filename);
    $myxls = $workbook->add_worksheet('Export');
    $myxls->set_column(0, 10, 30);

    /* Format style */
    $colnum = 0;
    $formatitle = $workbook->add_format();
    $formatitle->set_bold(3);
    $formatitle->set_size(16);
    $formathead = $workbook->add_format();
    $formathead->set_bold(1);
    $formathead->set_italic();
    $formathead->set_align('center');
    $formathead->set_bg_color('red');
    $formathead->set_fg_color('red');
    $formathead->set_size(14);
    $formathead->set_bottom(2);
    $formatrowodd = $workbook->add_format();
    $formatrowodd->set_bg_color('silver');
    $formatrowodd->set_fg_color('silver');
    $formatrowodd->set_bottom(2);
    $formatroweven = $workbook->add_format();
    $formatroweven->set_fg_color('white');
    $formatroweven->set_bg_color('white');
    $formatroweven->set_bottom(2);

    /* Xls Title */
    $myxls->write(0, 0, $table->title, $formatitle);

    $column = 0;
    $row = 2;

    /* Print table heading */
    foreach ($table->head as $h) {

        $myxls->write($row, $column, strip_tags($h), $formathead);
        $column++;
    }

    $row++;

    /* Print table body */
    foreach ($table->data as $data) {
        $column = 0;

        if ($row % 2!=0) {
            $formatrow = $formatrowodd;
        } else {
            $formatrow = $formatroweven;
        }

        foreach ($data as $d) {

            $myxls->write($row, $column, $d, $formatrow);
            $column++;
        }

        $row++;
    }

    $workbook->close();

}

/**
 * This function exports an input html table to PDF , using TCPDF library
 *
 * @param object $table Html table object
 */
function tlive_exportpdf($table) {

    global $CFG;

    /* Load tcpdf library */
     require_once("$CFG->libdir/pdflib.php");

    /*  create new PDF document */
    $pdf = new pdf;
    $pdf->setPrintHeader(false);
    $pdf->setPrintFooter(false);

    /* set font */
    $pdf->SetFont('helvetica',  'B',  16);
    /*  add a page */
    $pdf->AddPage();
    $title = $table->title;
    $pdf->Write(0,  $title,  '',  0,  'L',  true,  0,  false,  false,  0);
    $pdf->SetFont('freesans',  '',  8);

    $html = tlive_buildhtmltable($table);

    $pdf->writeHTML(utf8_encode($html),  true,  false,  false,  false,  '');
    $pdf->Output('export.pdf',  'D');

}

/**
 * This function build an html table using a Moodle table object as input
 *
 * @param object $table Html table object
 * @return string $html Html table
 */
function tlive_buildhtmltable($table) {

    $html="<p></p>
    <table border=\"1\">
    <tr style=\"font-size: large;\" bgcolor=\"#cc1111\">";

    foreach ($table->head as $h) {
        $html.="<td>$h</td>";
    }

    $html.='</tr></thead>
    <tbody>';
    $count=1;

    foreach ($table->data as $data) {

        if ($count % 2==0) {
            $color = "#bce7ff";
        } else {
            $color = "#ffffff";
        }

        $html.="<tr bgcolor=\"$color\">";

        foreach ($data as $d) {
            $html.="<td>$d</td>";
        }

        $html.="</tr>";

        $count++;
    }
    $html.="</table>";

    return $html;
}

/**
 * This function sort html table data by a field parsed through GET object
 *
 * @return object $sortparams Sort params
 */
function tlive_sort_tablebyfield() {

    $qrstr = $_SERVER['PHP_SELF']."?".$_SERVER['QUERY_STRING'];

    $sort = optional_param('sort', '', PARAM_TEXT);
    $order = optional_param('order', '', PARAM_TEXT);

    if (empty($sort) && empty($order)) {

        $sortfield = "startdate, enddate ";
        $sortdirection ="ASC";

    } else {
        /* clear old sort and order rule */
        $qrstr = str_replace('&sort='.$sort.'&order='.$order, '', $qrstr);

        $sortfield = $sort;
        $sortdirection = $order;
    }

    $newsortdirection = $sortdirection;
    $newsortdirection = $newsortdirection == 'ASC'? 'DESC':'ASC';

    $sortparams = new stdClass();
    /* querystring */
    $sortparams->qrStr = $qrstr;
    /* sort field */
    $sortparams->sort_field = $sortfield;
    /* actual sort direction */
    $sortparams->sort_direction = $sortdirection;
    /* next sort direction (after refresh) */
    $sortparams->new_sort_direction = $newsortdirection;

    return $sortparams;

}

/**
 * This function prints an html table that summarizes channel data
 *
 * @param string $xmlanswer XML answer from Teleskill Live server
 */
function tlive_displaychanneldata($xmlanswer) {

    $credits = tlive_readxmlanswer($xmlanswer, 'credits');
    $users = tlive_readxmlanswer($xmlanswer,  'users');
    $rooms = tlive_readxmlanswer($xmlanswer,  'rooms');

    $strcredits = ($credits) ? get_string('credits', 'tlive', $credits) : get_string('flatchannel', 'tlive');
    $strusers   = ($users)   ? $users : get_string('unlimited', 'tlive');
    $strrooms   = $rooms     ? get_string('rooms', 'tlive', $rooms) : get_string('unlimitedrooms', 'tlive');

    $table = new html_table();
    $table->tablealign = 'center';
    $table->head = array(get_string('header_credits', 'tlive'),
                         get_string('header_users', 'tlive'),
                         get_string('header_rooms', 'tlive'));
    $table->data[] = array($strcredits,  $strusers,  $strrooms);
    $table->align = array('left', 'left', 'left');
    $table->wrap = array(false, false, false);
    $table->size = array('33%', '33%', '34%');

    echo html_writer::table($table);
    print_string('channelloginfo',  'tlive');
}

/**
 * This function is called by cron plugin and checks for each room, if is necessary to update its statistics
 * @param int $daylimit threshold in days, beyond which is not necessary to update room statistics
 */
function tlive_cron_updatestats($daylimit) {

    global $DB;
    /* Don't update rooms that have finished from more than two days */

    $today = time();
    $diff = $today - $daylimit*3600*24;

    $query = 'SELECT * FROM {tlive} WHERE enddate>=? AND startdate<=?';

    if ($rooms = $DB->get_records_sql($query, array($diff, $today) ) ) {
        /* Check for each room */
        foreach ($rooms as $room) {

            if (! $course = $DB->get_record('course',  array('id' => $room->course))) {
                error('Course is misconfigured');
            }

            if (! $cm = get_coursemodule_from_instance('tlive',  $room->id,  $course->id)) {
                error('Course Module ID was incorrect');
            }

            /* Update rooms statistics and activities completion */
            tlive_updatestats($room, $course, $cm);

        }

    }

}

/**
 * This function is called by cron plugin and checks, if there are new recs then save them into moodle file repository
 *
 */
function tlive_cron_checkrecs() {

    global $DB, $USER;

    /* XML request to check new recs */
    $xmlanswer ='<?xml version="1.0" encoding="UTF-8"?>
        <ews>
        <errorcode>0</errorcode>
        <errormessage></errormessage>
        <recs>
        <rec id="1" title="Video 1" roomid="23251" status="1" date="2013-09-02 10:00:00" duration="3600">
        <file id="1" type="1" mode="2" src="http://rec.teleskill.it/rec-video/sied_2013_09_10_3363_flv.zip" />
        </rec>
        <rec id="2" title="Video 2" roomid="23436" status="1" date="2013-09-02 10:00:00" duration="3600">
        <file id="2" type="1" mode="2" src="http://rec.teleskill.it/rec-video/flex_2013_06_26_3197_mp4.zip" />
        <file id="3" type="1" mode="2" src="http://rec.teleskill.it/rec-video/escueladeinteligencia_2013_08_29_3341_mp4.zip" />
        </rec>
        </recs>
        </ews>';

    $xmlstructure = simplexml_load_string($xmlanswer);

    if ($recs = $xmlstructure->recs) {

        $recsfiles = array();

        foreach ($recs->rec as $rec) {

            $query = 'SELECT id,course FROM {tlive} WHERE roomid='.$rec->attributes()->roomid;
            $result = $DB->get_record_sql($query);

            /* if rec is available save files*/
            if ($rec->attributes()->status==1) {

                if ($files = $rec->file) {

                    foreach ($files as $file) {

                        $filesrc = (string) $file->attributes()->src;
                        $filename = pathinfo($filesrc, PATHINFO_BASENAME);

                        /* verify if this rec is not already saved into repository */
                        $query = "SELECT * FROM {tlive_repository}
                        WHERE recid=".$rec->attributes()->id." AND fileid='".$file->attributes()->id."'";

                        if (!$DB->get_record_sql($query)) {

                            $recobj = new stdClass();
                            $recobj->title = (string)$rec->attributes()->title;
                            $recobj->filename = (string) $filename;
                            $recobj->src = (string)$filesrc;
                            $recobj->timecreated = strtotime($rec->attributes()->date);
                            $recobj->courseid = (int)$result->course;
                            $recobj->recid = (int)$rec->attributes()->id;
                            $recobj->fileid = (int)$file->attributes()->id;

                            /* Call function to save and store file */
                            tlive_save_rec($recobj);

                        }

                    }
                }
            }
        }
    }
}

/**
 * This function is called by tlive_cron_checkrecs and download and store current new rec
 * @param object $rec Current new rec
 */
function tlive_save_rec($rec) {

    global $CFG, $DB;

    $tmppath = $CFG->dataroot.'/temp/';
    $filename = $rec->filename;

    /* Create curl object to download file rec */
    $c = new curl;
    $result = $c->download_one($rec->src, null, array('filepath' => $tmppath.$filename, 'timeout' => 400));

    if (!$result) {

         throw new moodle_exception('errorwhiledownload', 'repository', '', $result);

    }

    /* extract file extension */
    $ext = pathinfo($tmppath.$filename, PATHINFO_EXTENSION);

    /* if file is a zip, unzip it and save uncompressed file */
    if ($ext=='zip') {
        $zip = new ZipArchive;
        $zipfile = $zip->open($tmppath.$filename);

        if (!$zipfile) {

            print_error('errorunzipfile', 'tlive');

        } else {

            $zip->extractTo($tmppath);

            /* use uncompressed file */
            $filename = $zip->getNameIndex(0);
            $zip->close();

            /* delete temp zip file */
            fulldelete($tmppath.$rec->filename);
            /* update filename object's property */
            $rec->filename = $filename;
        }

    }

    $fs = get_file_storage();
    $coursecontext = context_course::instance($rec->courseid);

    $filerecord = array('contextid'=>$coursecontext->id, 'component'=>'mod_tlive', 'filearea'=>'recs',
             'itemid'=>0, 'filepath'=>'/', 'filename'=>$filename,
             'timecreated'=>$rec->timecreated , 'timemodified'=>time());

    if (!$result = $fs->create_file_from_pathname($filerecord, $tmppath.$filename)) {
        throw new moodle_exception('errorwhiledownload', 'repository', '', $result);
    }

    $rec->size  =  filesize($tmppath.$filename);
    /* save file reference in db */
    $DB->insert_record('tlive_repository', $rec, false);
    /* delete tmp file */
    fulldelete($tmppath.$filename);

}

/**
 * This function is called by tlive_cron and checks if current time t€[tin,tout]
 * @param int $tin left bound of the interval
 * @param int $tout right bound of the interval
 * @param int $t current instant
 * @return boolean
 */
function tlive_checktime($tin, $tout, $t) {

    /* ex: (20,23,22) */
    if ($tin<$tout) {
        return ($t>=$tin && $t<$tout);
        /* ex: (23,5,3) if 3 is not in (5,23) then is included between 23 and 5 */
    } else {
        return !($t>=$tout && $t<$tin);

    }
}

/**
 * This function is called by events.php and format user timestamp to ISO 8601
 * @param int $date timestamp format
 * @return string $aux date ISO8601
 */
function tlive_get_isodatetime($date) {

    $year = userdate($date, '%Y');
    $month = userdate($date, '%m');
    $day = str_pad(userdate($date, '%d'), 2, '0', STR_PAD_LEFT);
    $hour = userdate($date, '%H');
    $minutes = userdate($date, '%M');
    $seconds = userdate($date, '%S');

    $usertimeoffset = get_user_timezone_offset();

    /* if time zone is default (local time server) time offset is 0*/
    if ($usertimeoffset == 99) {

        $usertimeoffset = date('Z')/3600;

    }

    /* compute daylight saving offset */
    $dst = dst_offset_on($date)/3600;
    $usertimeoffset = $usertimeoffset + $dst;

    $usertimeoffset = str_pad($usertimeoffset, 2, 0, STR_PAD_LEFT);

    /* trim day used because userdate() function inserts a space before 2-digit days (moodle bug) MDL-11601*/
    $aux = $year.'-'.$month.'-'.trim($day).'T'.$hour.':'.$minutes.':'.$seconds.'+'.$usertimeoffset.':00';

    return $aux;

}