<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Library of interface functions and constants.
 *
 * @package     mod_tlive
 * @copyright   Teleskill Italia S.r.l
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

define('TLIVE_INCLUDE_TEST',  1);
define('TLIVE_NEWLINE',  "\r\n");
define('TLIVE_INDENT',  '    '); 
/* TABS */
define('TLIVE_NOTAB',  0);
define('TLIVE_TAB1',  1);
define('TLIVE_TAB2',  2);
/* PAGES
 * pages of tab 1
 * no pages foreseen for first tab
 * pages of tab 2 */
define('TLIVE_TAB2_PAGE1',  '1');
define('TLIVE_TAB2_PAGE1NAME',  'tab2page1');
define('TLIVE_TAB2_PAGE2',  '2');
define('TLIVE_TAB2_PAGE2NAME',  'tab2page2');

define('TLIVE_REC_FORMAT_FLV',  1);
define('TLIVE_REC_FORMAT_MP4',  2);
define('TLIVE_REC_FORMAT_WEBM',  3);

defined('MOODLE_INTERNAL') || die();

require_once($CFG->dirroot.'/mod/tlive/custom_event/event_availability.php');
require_once($CFG->dirroot.'/calendar/lib.php');

/**
 * Return if the plugin supports $feature.
 *
 * @param string $feature Constant representing the feature.
 * @return true | null True if the feature is supported, null otherwise.
 */
function tlive_supports($feature) {
    switch ($feature) {
        case FEATURE_GROUPS:
            return false;
        case FEATURE_GROUPINGS:
            return false;
        case FEATURE_MOD_INTRO:
            return true;
        case FEATURE_COMPLETION_TRACKS_VIEWS:
            return false;
        case FEATURE_GRADE_HAS_GRADE:
            return false;
        case FEATURE_GRADE_OUTCOMES:
            return false;
        case FEATURE_BACKUP_MOODLE2:
            return true;
        case FEATURE_COMPLETION_HAS_RULES:
            return true;
        case FEATURE_SHOW_DESCRIPTION:        
        	return true;
        default:
            return null;
    }
}

/**
 * Saves a new instance of the mod_tlive into the database.
 *
 * Given an object containing all the necessary data, (defined by the form
 * in mod_form.php) this function will create a new instance and return the id
 * number of the instance.
 *
 * @param object $tlive An object from the form.
 * @param mod_tlive_mod_form $mform The form.
 * @return int The id of the newly inserted record.
 */

function tlive_add_instance($tlive, $mform = null) {
    global $CFG, $DB;

    $checkboxes = array('emailconfman',  'opentoblank');
    foreach ($checkboxes as $checkbox) {
        if (empty($tlive->{$checkbox})) {
            $tlive->{$checkbox} = 0;
        }
    }

   //Save multiple confmanid with implode into database
    $tlive->confmanid = implode(',', $tlive->confmanid);
    

    $tlive->timecreated = time();

    /* It rebuilds complete datetime format
     * Start date temporary storage */
    $startdate = $tlive->startdate;
    $tlive->startdate = $tlive->startdate + ($tlive->starthour) * 3600 + ($tlive->startminute) * 60;
    $tlive->enddate =  $startdate + ($tlive->endhour) * 3600 + ($tlive->endminute) * 60;

    /* Check availability places */
    $maxusers = $CFG->tlive_maxusers;

     /* if scheduling is disabled, don't use startdate and enddate*/
    if (!$CFG->tlive_disabledate) {

        $success = event_availability($maxusers, $tlive->users, $tlive->startdate, $tlive->enddate);
        if (!$success) {
            print_error('places_not_available', 'tlive');
        }
    }

    $xmlanswer = tlive_createroom($tlive, $mform);
    if (!tlive_readxmlanswer($xmlanswer,  'errorcode')) {
        $tlive->roomid = tlive_readxmlanswer($xmlanswer,  'roomid');
    }
    if (!isset($tlive->roomid)) {
        print_error('roomcreationfailed', 'tlive');
    }

    if ($tliveid = $DB->insert_record('tlive',  $tlive)) {

        if (!empty($tlive->roomid)) {

            if ($tlive->emailconfman ) {

                //Explode conference manager id into array for sending email to all manager
                $confmanagerid = explode(',', $tlive->confmanid);

                //Loop each conference manager and send individual email to all
                foreach ($confmanagerid as $managerid) {

                    tlive_sendemail($tlive, $managerid ,  'add');

                  }
                
            }
        }

        if (!$CFG->tlive_disabledate) {
            tlive_addcalendarevent($tlive, $tliveid);
        }
    }
    return $tliveid;
}

/**
 * Updates an instance of the mod_tlive in the database.
 *
 * Given an object containing all the necessary data (defined in mod_form.php),
 * this function will update an existing instance with new data.
 *
 * @param object $tlive An object from the form in mod_form.php.
 * @param mod_tlive_mod_form $mform The form.
 * @return bool True if successful, false otherwise.
 */
function tlive_update_instance($tlive, $mform = null) {
    global $CFG, $DB;

    $checkboxes = array('emailconfman',  'opentoblank');
    foreach ($checkboxes as $checkbox) {
        if (empty($tlive->{$checkbox})) {
            $tlive->{$checkbox} = 0;
        }
    }
    /* if ($tlive->startdate == $tlive->enddate) {
           $tlive->enddate += 3600;
       } */
    $tlive->timemodified = time();
    $tlive->id = $tlive->instance;

    /* It rebuilds complete datetime format
     * Start date temporary storage */

     $startdate = $tlive->startdate;

     /* set startdate and enddate to 0, if scheduling is disabled */
    if($CFG->tlive_disabledate) {

        $startdate = 0;
    }

    //Update multiple confmanid with implode into database
    $tlive->confmanid = implode(',', $tlive->confmanid);

    $tlive->startdate = $startdate +  ($tlive->starthour)*3600 + ($tlive->startminute)*60;
    $tlive->enddate =  $startdate +  ($tlive->endhour)*3600 + ($tlive->endminute)*60;

    $oldtlive = $DB->get_record('tlive',  array('id' => $tlive->id));

    /* Check availability places */
    $maxusers = $CFG->tlive_maxusers;

    /* if date scheduling is enabled */
    if (!$CFG->tlive_disabledate) {
        $success = event_availability($maxusers, $tlive->users, $tlive->startdate, $tlive->enddate,  $tlive->id);

        if (!$success) {
            print_error('places_not_available', 'tlive');
        }

    }

    if (empty($oldtlive->roomid)) {
        $xmlanswer = tlive_createroom($tlive, $mform);
        if (!tlive_readxmlanswer($xmlanswer,  'errorcode')) {
            $tlive->roomid = tlive_readxmlanswer($xmlanswer,  'roomid');
        }
    } else {
        $tlive->roomid = $oldtlive->roomid;
        $xmlanswer = tlive_modifyroom($tlive, $mform);
        if (tlive_readxmlanswer($xmlanswer,  'errorcode')) {
            print_error('roomcreationfailed', 'tlive');

        }
    }

    if ($tliveid = $DB->update_record('tlive',  $tlive)) {

        if (!$CFG->tlive_disabledate) {

            tlive_updatecalendarevent($tlive);
        }

        if (!empty($tlive->roomid)) {
            /* startdate */
            $oldstartdate = $oldtlive->startdate;
            $newstartdate = $tlive->startdate;

            /*  conference manager id , explode all conference id for email notify*/
            $oldconfman = explode(',', $oldtlive->confmanid);
            $newconfman = explode(',', $tlive->confmanid);

            // Sorting the array elements for checking they are equal or not
            sort($oldconfman);
            sort($newconfman);

            // keep old and new conference manager id into array for sending email
            $oldmanagerid = explode(',', $oldtlive->confmanid);
            $newmanagerid = explode(',', $tlive->confmanid);

            /* email conference manager */
            $oldemailconfman = $oldtlive->emailconfman;
            $newemailconfman = empty($tlive->emailconfman) ? 0 : 1;

            if ($oldconfman != $newconfman) { /* If conference manager is changed */

                if ($oldemailconfman) { /* And if it sent a mail to notify the assigned role */
                    /* Send a new mail to notify the removal from role*/

                    //Find the list of removed conference manager

                    $rmmanagerid = array_diff($oldmanagerid,$newmanagerid);
                    
                    //Loop each removed conference manager and send individual email to all
                    foreach ($rmmanagerid as $managerid) {

                        tlive_sendemail($oldtlive, $managerid, 'retire');
                    } 

                }

                if ($newemailconfman) {

                    //Find the list of added new conference manager

                    $addmanagerid = array_diff($newmanagerid,$oldmanagerid);
                    
                    //Loop each added new conference manager and send individual email to all
                    foreach ($addmanagerid as $managerid) {

                        tlive_sendemail($oldtlive, $managerid, 'add');
                    }

                }

            } else {

                /* If conference manager is the same,  send an e-mail to notify him */
                if ($newemailconfman) {

                    if ($oldemailconfman) {

                        if ($oldstartdate != $newstartdate) {  /* If startdate is changed */

                            /* if date scheduling is enabled */
                            if (!$CFG->tlive_disabledate) {

                                 //Loop all same old conference manager and send individual email to all
                                foreach ($oldmanagerid as $managerid) {

                                    tlive_sendemail($tlive, $managerid, 'update');

                                }

                            }

                        }

                    } else {

                        //Loop all old same conference manager and send individual email to all
                        foreach ($oldmanagerid as $managerid) {

                            tlive_sendemail($tlive, $managerid, 'add');

                        }

                    }
                }
            }
        }
    }
    return $tliveid;
}

/**
 * Removes an instance of the mod_tlive from the database.
 *
 * @param int $id Id of the module instance.
 * @return bool True if successful, false on failure.
 */
function tlive_delete_instance($id) {
    global $DB;

    if (! $tlive = $DB->get_record('tlive',  array('id' => $id))) {
        return false;
    }

    $result = true;

    //Get record from database for email notify
    $oldtlive = $DB->get_record('tlive',  array('id' => $id));

    // keep all old conference manager id into array for sending email
    $oldmanagerid = explode(',', $oldtlive->confmanid);

    /* Delete any dependent records here
     * room for an instance is supposed to be never more than one */
    $xmlanswer = tlive_deleteroom($tlive);
    /* if ($tlive->emailconfman && ($tlive->startdate > time())) { */
    if ($tlive->emailconfman) {
        /* se il conference manager era stato avvertito,  disdici l'incarico */

        //Loop all old conference manager and send email to all for removed instance
        foreach ($oldmanagerid as $managerid) {

            tlive_sendemail($tlive, $managerid, 'del');

        } 

    }

    if (! $DB->delete_records('tlive',  array('id' => $tlive->id))) {
        $result = false;
    }

    return $result;
}

/**
 * Function that build XML request type heading (add,edit,delete room...)
 *
 * @param int $type Request type
 * @return string $xmlrequest XML request
 */
function teleskill_beginrequest($type) {
    $xmlrequest = '<?xml version="1.0" encoding="UTF-8"?>'.TLIVE_NEWLINE;
    $xmllang = (current_language() == 'it_utf8') ? 'IT' : 'EN';
    $xmlrequest .= '<ews type="'.$type.'" lang="'.$xmllang.'">'.TLIVE_NEWLINE;
    return $xmlrequest;
}

/**
 * Function that sends http request to Teleskill Live server, using fsock
 *
 * @param int $xmlrequest Request type
 * @return string $xmlanswer XML answer
 */
function tlive_getanswerfrom($xmlrequest) {
    /* custom function file_get_contents() that sends data through POST request */
    global $CFG;
    $xmlanswer = false;

    $url = $CFG->tlive_serverhost;

    $headers = array(
        "Content-type: text/xml",
        "Content-length: " . strlen($xmlrequest),
        "Connection: close",
    );

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL,$url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $xmlrequest);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

    $bypassArray = explode(',', $CFG->proxybypass);
    foreach ($bypassArray as $key => $value) {
        $bypassArray[$key] = trim($value);
    }

    if($CFG->proxyhost != '' && !in_array('asp.teleskill.it', $bypassArray)) {
        print_r('proxy' . '<br>');
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_HTTPPROXYTUNNEL , 1);
        curl_setopt($ch, CURLOPT_PROXY, $CFG->proxyhost);
        curl_setopt($ch, CURLOPT_PROXYPORT, $CFG->proxyport);
        if($CFG->proxyuser != '') {
            curl_setopt($ch, CURLOPT_PROXYUSERPWD, "$CFG->proxyuser:$CFG->proxypassword");
        }
    }

    $xmlanswer = curl_exec($ch); 

    if(curl_errno($ch)) {
        print curl_error($ch);
    } 
     
    curl_close($ch);

    if (($CFG->debug >= DEBUG_ALL) && $CFG->debugdisplay) {
        echo '$xmlrequest = <textarea cols="90" rows="10">' . $xmlrequest . '</textarea><br />';

        echo '$xmlanswer = <textarea cols="90" rows="10">' . $xmlanswer . '</textarea><br />';
        print_string('reducedebuglevel', 'tlive');
    }

    return $xmlanswer;
}


/**
 * Function that validates XML answer string
 *
 * @param string $xmlanswer Tlive object instance
 * @param string $tag Form object
 * @return string
 */
function tlive_readxmlanswer($xmlanswer,  $tag) {

    global $CFG;

    if (empty($xmlanswer)) {
        if ($CFG->debug < DEBUG_ALL) {
            $a = get_string('debugall',  'admin');
            $a = get_string('riseupdebuglevel',  'tlive',  $a);
        } else {
            $a = '';
        }
        return get_string('emptyxml', 'tlive',  $a);
    }

    $regexp = "|<$tag>(.*)</$tag>|";
    if (preg_match($regexp,  $xmlanswer,  $matches)) {
        return $matches[1]; /* it is a single value */
    }
}

/**
 * Function that send a request to create a videoconference room, through an XML
 *
 * @param object $tlive Tlive object instance
 * @param object $mform Form object
 * @return string $xml_answer XML answer from Teleskill Live server
 */
function tlive_createroom($tlive, $mform) {

    global $CFG, $COURSE;
    /*  notice: $tlive is passed as params because during
     * calling of add_instance, $tlive not exists as a global var
     */
    $xmlanswer = tlive_isroomavailable($tlive);
    if (tlive_readxmlanswer($xmlanswer,  'errorcode')) {
        /* The room is not available,  return the message */
        return $xmlanswer;
    } else {
        /* The room is available,  try to create it */
        $xmlrequest = teleskill_beginrequest('1');

        $xmlrequest .= TLIVE_INDENT.'<clientcode>'.$CFG->tlive_clientcode.'</clientcode>'.TLIVE_NEWLINE;

        $startdate = '';
        $enddate = '';

        if (!$CFG->tlive_disabledate) {

            $startdate = gmdate("Y-m-d H:i:s",  $tlive->startdate);
            $enddate = gmdate("Y-m-d H:i:s",  $tlive->enddate);
        }

        /* if scheduling isn't disabled, use startdate and enddate */

        $xmlrequest .= TLIVE_INDENT.'<startdate>'.$startdate.'</startdate>'.TLIVE_NEWLINE;
        $xmlrequest .= TLIVE_INDENT.'<enddate>'.$enddate.'</enddate>'.TLIVE_NEWLINE;


        $xmlrequest .= TLIVE_INDENT.'<title>'.$COURSE->shortname.' - '.$tlive->name.'</title>'.TLIVE_NEWLINE;
        /* if ($tlive->intro) { */
        /* At the moment, not include intro description */
        /* $xmlrequest .= TLIVE_INDENT.'<descr>'.$tlive->intro.'</descr>'.TLIVE_NEWLINE; */
        /* } */

        $xmlrequest .= TLIVE_INDENT.'<users>'.$tlive->users.'</users>'.TLIVE_NEWLINE;
        $avquality = (isset($tlive->avquality) && $tlive->avquality > 0) ? $tlive->avquality : '';    
    	$xmlrequest .= TLIVE_INDENT.'<avquality>'.$avquality.'</avquality>'.TLIVE_NEWLINE;	
        
        $context = context_module::instance($tlive->coursemodule);

        /* logo image */
        if ($filename = $mform->get_new_filename('logo')) {

            $itemid = file_get_submitted_draft_itemid('logo');

            $file = $mform->save_stored_file('logo', $context->id, 'mod_tlive', 'rooms',
            $itemid, '/', $filename);

            $basepath = $CFG->wwwroot.'/pluginfile.php/'.$context->id.'/mod_tlive/rooms/'.$itemid;
            $xmlrequest .= TLIVE_INDENT.'<logo>'.$basepath.'/'.$filename.'</logo>'.TLIVE_NEWLINE;

        }


        /* banner image */
         if ($filename = $mform->get_new_filename('banner')) {

            $itemid = file_get_submitted_draft_itemid('banner');

            $file = $mform->save_stored_file('banner', $context->id, 'mod_tlive', 'rooms',
                $itemid, '/', $filename);

            $basepath = $CFG->wwwroot.'/pluginfile.php/'.$context->id.'/mod_tlive/rooms/'.$itemid;
            $xmlrequest .= TLIVE_INDENT.'<banner>'.$basepath.'/'.$filename.'</banner>'.TLIVE_NEWLINE;

        }

        $xmlrequest .= '</ews>';

        /* return XML answer string that must be analyzed from receiving */
        $xmlanswer = tlive_getanswerfrom($xmlrequest);
        if (!tlive_readxmlanswer($xmlanswer,  'errorcode')) {
            $tlive->roomid = tlive_readxmlanswer($xmlanswer,  'roomid');
        }

        return $xmlanswer;
    }
}

/**
 * Function that check room availability
 *
 * @param object $tlive Tlive object instance
 * @return string XML answer
 */
function tlive_isroomavailable($tlive) {
    global $CFG;

    /*  notice: $tlive is passed as params because during
     * calling of create_instance, $tlive not exists as a global var
     */

    $xmlrequest = teleskill_beginrequest('5');

    $xmlrequest .= TLIVE_INDENT.'<clientcode>'.$CFG->tlive_clientcode.'</clientcode>'.TLIVE_NEWLINE;

    /* if scheduling isn't disabled, use startdate and enddate */
    if(!$CFG->tlive_disabledate) {
        $xmlrequest .= TLIVE_INDENT.'<startdate>'.gmdate("Y-m-d H:i:s",  $tlive->startdate).'</startdate>'.TLIVE_NEWLINE;
        $xmlrequest .= TLIVE_INDENT.'<enddate>'.gmdate("Y-m-d H:i:s",  $tlive->enddate).'</enddate>'.TLIVE_NEWLINE;
    }
    $xmlrequest .= TLIVE_INDENT.'<users>'.$tlive->users.'</users>'.TLIVE_NEWLINE;
    $xmlrequest .= '</ews>';

    return tlive_getanswerfrom($xmlrequest);
}

/**
 * Function that sends an e-mail to user
 *
 * @param object $tlive Tlive object instance
 * @param string $action Type of message (add user,retire..)
 * @param int $managerid Manager ID
 */
function tlive_sendemail($tlive, $managerid,  $action) {
    
    global $CFG,  $USER,  $COURSE,  $DB;

    $a = new stdClass();
    $a->courseurl = $CFG->wwwroot.'/course/view.php?id='.$COURSE->id;
    $a->coursename = $COURSE->fullname;
    $a->conferencename = $tlive->name;
    $a->timestart = userdate($tlive->startdate);

    switch ($CFG->tlive_disabledate) {

        case 0:
            $string = 'emailbody_'.$action;
            break;
        case 1:
            $string = 'emailbody_'.$action.'_nodate';
            break;

    }

    $htmlbody = get_string($string, 'tlive',$a);
    $subject= get_string('emailsubject', 'tlive',  $a);
    $textbody = strip_tags($htmlbody);
    $recipient = $DB->get_record('user',  array('id' => $managerid));
    $from = $DB->get_record('user',  array('id' => $USER->id));

    if (!email_to_user($recipient,  $from,  $subject,  $textbody,  $htmlbody)) {
        $a = new stdClass();
        $a->roomcreator = fullname($from);
        $a->conferencemanager = fullname($recipient);
        print_string('emailnotsent', 'tlive',  $a);
    }
}

/**
 * This function add new calendar event for Tlive instance
 *
 * @param object $tlive tlive instance
 * @param object $tliveid tlive id new instance
 */
function tlive_addcalendarevent($tlive, $tliveid) {

    $event = new stdClass();
    $event->name = $tlive->name;
    $event->description = $tlive->intro;
    $event->courseid = $tlive->course;
    $event->groupid      = 0;
    $event->userid       = 0;
    $event->modulename   = 'tlive';
    $event->eventtype    = 'course';
    $event->instance     = $tliveid;
    $event->timestart    = $tlive->startdate;
    $event->timeduration = $tlive->enddate - $tlive->startdate;
    $event->visible      = instance_is_visible('tlive', $tlive);

    calendar_event::create($event);

}

/**
 * Function that build the XML request to modify room data
 *
 * @param object $tlive Tlive object instance
 * @param object $mform Form object
 * @return string XML answer
 */
function tlive_modifyroom($tlive, $mform) {
    global $CFG, $COURSE;

    $xmlrequest = teleskill_beginrequest('6');

    $xmlrequest .= TLIVE_INDENT.'<clientcode>'.$CFG->tlive_clientcode.'</clientcode>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<roomid>'.$tlive->roomid.'</roomid>'.TLIVE_NEWLINE;
    /* if scheduling isn't disabled, use startdate and enddate */

    $startdate = '';
    $enddate = '';

    if (!$CFG->tlive_disabledate) {

        $startdate = gmdate("Y-m-d H:i:s",  $tlive->startdate);
        $enddate = gmdate("Y-m-d H:i:s",  $tlive->enddate);
    }


    $xmlrequest .= TLIVE_INDENT.'<startdate>'.$startdate.'</startdate>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<enddate>'.$enddate.'</enddate>'.TLIVE_NEWLINE;

    $xmlrequest .= TLIVE_INDENT.'<title>'.$COURSE->shortname.' - '.$tlive->name.'</title>'.TLIVE_NEWLINE;
    /* if ($tlive->intro) {
         * $xmlrequest .= TLIVE_INDENT.'<descr>'.$tlive->intro.'</descr>'.TLIVE_NEWLINE;
     }
     */

    $xmlrequest .= TLIVE_INDENT.'<users>'.$tlive->users.'</users>'.TLIVE_NEWLINE;    
    
    $avquality = (isset($tlive->avquality) && $tlive->avquality > 0) ? $tlive->avquality : '';
    
    $xmlrequest .= TLIVE_INDENT.'<avquality>'.$avquality.'</avquality>'.TLIVE_NEWLINE;	
    
    $context = context_module::instance($tlive->coursemodule);

    /* logo image */
    if ($filename = $mform->get_new_filename('logo')) {

        $itemid = file_get_submitted_draft_itemid('logo');

        $file = $mform->save_stored_file('logo', $context->id, 'mod_tlive', 'rooms',
        $itemid, '/', $filename,true);

        $basepath = $CFG->wwwroot.'/pluginfile.php/'.$context->id.'/mod_tlive/rooms/'.$itemid;
        $xmlrequest .= TLIVE_INDENT.'<logo>'.$basepath.'/'.$filename.'</logo>'.TLIVE_NEWLINE;

    }


    /* banner image */
     if ($filename = $mform->get_new_filename('banner')) {

        $itemid = file_get_submitted_draft_itemid('banner');

        $file = $mform->save_stored_file('banner', $context->id, 'mod_tlive', 'rooms',
            $itemid, '/', $filename,true);

        $basepath = $CFG->wwwroot.'/pluginfile.php/'.$context->id.'/mod_tlive/rooms/'.$itemid;
        $xmlrequest .= TLIVE_INDENT.'<banner>'.$basepath.'/'.$filename.'</banner>'.TLIVE_NEWLINE;

    }

    $xmlrequest .= '</ews>';

    return tlive_getanswerfrom($xmlrequest);
}

/**
 * This function update calendar event for current Tlive instance
 *
 * @param object $tlive tlive instance
 */
function tlive_updatecalendarevent($tlive) {

    global $DB;

    if ($eventrecord = $DB->get_record('event', array('modulename'=>'tlive', 'instance'=>$tlive->id)) ) {

        $event = calendar_event::load($eventrecord->id);
        $updevent = new stdClass();
        $updevent->name = $tlive->name;
        $updevent->description = $tlive->intro;
        $updevent->timestart    = $tlive->startdate;
        $updevent->timeduration = $tlive->enddate - $tlive->startdate;

        $event->update($updevent);

    }

}

/**
 * Function that build the XML request to delete room
 *
 * @param object $tlive Tlive object instance
 * @return string XML answer
 */
function tlive_deleteroom($tlive) {
    global $CFG;

    /*  notice: $tlive is passed as params because during
     * calling of delete_instance, $tlive not exists as a global var
     */

    $xmlrequest = teleskill_beginrequest('2');

    $xmlrequest .= TLIVE_INDENT.'<clientcode>'.$CFG->tlive_clientcode.'</clientcode>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<roomid>'.$tlive->roomid.'</roomid>'.TLIVE_NEWLINE;
    $xmlrequest .= '</ews>';

    return tlive_getanswerfrom($xmlrequest);
}

/**
 * This function checks if a user
 * achieves the conditions to complete an activity, respect to a selected rule.
 *
 * @param object $course Course
 * @param object $cm Course-module
 * @param int $userid User ID
 * @param bool $type Type of comparison (or/and; can be used as return value if no conditions)
 * @return bool True if completed, false if not, $type if conditions not set.
 */
function tlive_get_completion_state($course, $cm, $userid, $type) {

    global $CFG, $DB;

    /* Get  details */
    if (!($tlive=$DB->get_record('tlive', array('id'=>$cm->instance)))) {
        throw new Exception("Can't find tlive {$cm->instance}");
    }

    /* Default return value */
    $result = $type;

    /* If enabled, checks activity completion  verifica il completamento del modulo in base ai minuti di frequenza */
    if ($tlive->completiontlive) {
        /* Compute total time of user attendance  */
        $duration = $DB->get_field_sql('SELECT COALESCE(SUM(uls.duration), 0) as duration FROM {tlive_lesson_user_sessions} as uls
            INNER JOIN {tlive_user_sessions} as us ON uls.usersessionid=us.usersessionid
            INNER JOIN {tlive_rooms_sessions} as rs ON us.roomsessionid=rs.roomsessionid
            WHERE us.iduser=? AND rs.roomid=? AND us.iduser<>?', array($userid, $tlive->roomid, $tlive->confmanid));

        /* if attendance>=min_attendance, user has completed activity */
        if ($duration && ($duration/60)>=$tlive->completiontlive) {
            $result = true;
            //add_to_log($course->id,  "tlive",  "completion",  "view.php?id=$cm->id", $tlive->name, $cm->id, $userid);
        } else {
            $result = false;
        }
    }

    if ($tlive->completiontlivepoll) {
        /* Count total sum of correct answers */
        $query = 'SELECT COALESCE(SUM(pa.correct), 0) as tot_correct FROM {tlive_users_poll_answers} as upa
        INNER JOIN {tlive_user_sessions} as us ON upa.usersessionid=us.usersessionid
        INNER JOIN {tlive_polls_answers} as pa ON upa.answerid=pa.answerid
        INNER JOIN {tlive_rooms_sessions} as rs ON us.roomsessionid=rs.roomsessionid
        WHERE rs.roomid=? AND us.iduser=? AND pa.correct=1 AND us.iduser<>?';
        $totcorrect = $DB->get_field_sql($query, array($tlive->roomid, $userid, $tlive->confmanid));

        /* if totcorrect>=min_correct_answers, user has completed activity */
        if ($totcorrect>=$tlive->completiontlivepoll) {
            $result = true;
            //add_to_log($course->id,  "tlive",  "completion",  "view.php?id=$cm->id", $tlive->name, $cm->id, $userid);
        } else {
            $result = false;
        }

    }

    if ($tlive->completiontliveviewsubmit) {

        /* Check if user has participant role and he has logged room at least once */
        $context = context_course::instance($course->id);

        /* query following moodle role
        *$query = "SELECT * FROM mdl_log as l
                    INNER JOIN mdl_role_assignments as r ON l.userid=r.userid
                    WHERE r.contextid=? AND r.userid=? AND l.`action`='join' AND r.roleid=5"; */

        /* query following tlive role */
        $query = "SELECT * FROM {logstore_standard_log}
                   WHERE component='mod_tlive' AND objectid=? AND userid=? AND action='join' AND userid <> ?";
          

        if ($res = $DB->get_records_sql($query, array($tlive->id, $userid, $tlive->confmanid))) {
            $result = true;
            //add_to_log($course->id,  "tlive",  "completion",  "view.php?id=$cm->id", $tlive->name, $cm->id, $userid);
        } else {
            $result=false;
        }

    }
        return $result;
}

/**
 * Function used by repository plugin to send a file requested by context and file area
 *
 * @param object $course course instance
 * @param object $cm course module instance
 * @param object $context instance context
 * @param string $filearea file area
 * @param array $args arguments
 * @param boolean $forcedownload force download file
 * @param array $options options
 */
function tlive_pluginfile($course, $cm, $context, $filearea, $args, $forcedownload, array $options=array()) {

    $filepath='/';
    $itemid = array_shift($args);
    $filename = array_pop($args);
    $fs = get_file_storage();
    $file = $fs->get_file($context->id, 'mod_tlive', $filearea, $itemid, $filepath, $filename);
    if (!$file) {
        return false; // The file does not exist.
    }

    send_stored_file($file, 86400, 0, $forcedownload, $options);

}

/**
 * This function returns if a scale is being used by one tlive
 * if it has support for grading and scales. Commented code should be
 * modified if necessary. See forum,  glossary or journal modules
 * as reference.
 *
 * @param int $tliveid ID of an instance of this module
 * @param int $scaleid
 * @return boolean True if the scale is used by the tlive
 */
function tlive_scale_used($tliveid,  $scaleid) {
    $return = false;
    /* $rec = get_record("tlive", "id", "$tliveid", "scale", "-$scaleid");
    if (!empty($rec) && !empty($scaleid)) {
       $return = true;
    } */
    return $return;
}

/**
 * Checks if scale is being used by any instance of tlive.
 * This function was added in 1.9
 *
 * This is used to find out if scale used anywhere
 * @param int $scaleid
 * @return boolean True if the scale is used by any tlive
 */
function tlive_scale_used_anywhere($scaleid) {
    /*if ($scaleid and record_exists('tlive',  'grade',  -$scaleid)) {
        return true;
    } else {
        return false;
    }*/
    return false;
}

/**
 * Return a small object with summary information about what a
 * user has done with a given particular instance of this module
 * Used for user activity reports.
 *
 * @param object $course Course object
 * @param object $user User object
 * @param object $mod
 * @param object $tlive Tlive instance
 * @return object
 */
function tlive_user_outline($course,  $user,  $mod,  $tlive) {
    return null;
}


/**
 * Print a detailed representation of what a user has done with
 * a given particular instance of this module,  for user activity reports.
 *
 * @param object $course Course object
 * @param object $user User object
 * @param object $mod
 * @param object $tlive
 * @return boolean
 */
function tlive_user_complete($course,  $user,  $mod,  $tlive) {
    return true;
}


/**
 * Given a course and a time,  this module should find recent activity
 * that has occurred in tlive activities and print it out.
 * Return true if there was output,  or false is there was none.
 *
 * @param object $course Course object
 * @param boolean $isteacher True if is a teacher, false otherwise
 * @param boolean $timestart
 * @return boolean
 */
function tlive_print_recent_activity($course,  $isteacher,  $timestart) {
    return false;  /*  True if anything was printed,  otherwise false */
}

/**
 * Must return an array of user records (all data) who are participants
 * for a given instance of tlive. Must include every user involved
 * in the instance,  independient of his role (student,  teacher,  admin...)
 * See other modules as example.
 *
 * @param int $tliveid ID of an instance of this module
 * @return mixed boolean/array of students
 */
function tlive_get_participants($tliveid) {
    return false;
}

/**
 * Execute post-install custom actions for the module
 * This function was added in 1.9
 *
 * @return boolean true if success,  false on error
 */
function tlive_install() {
    return true;
}
