<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Local functions file of tlive module
 *
 * @package     mod_tlive
 * @copyright   Teleskill Italia S.r.l
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
*/

require_once("$CFG->dirroot/mod/tlive/lib.php");

/**
 * This function builds XML request to get channel data
 *
 * @return string  XML answer returned by Teleskill Live
*/

function tlive_channeldatarequest() {
    global $CFG;

    $xmlrequest = teleskill_beginrequest('11');

    $xmlrequest .= TLIVE_INDENT.'<clientcode>'.$CFG->tlive_clientcode.'</clientcode>'.TLIVE_NEWLINE;
    $xmlrequest .= '</ews>';

    return tlive_getanswerfrom($xmlrequest);
}

/**
 * This function builds XML request to enable user access
 *
 * @param  object $tlive Tlive instance object
 * @return string  XML answer returned by Teleskill Live
 */
function tlive_enableuser($tlive) {

    global $CFG,  $USER , $cm;

    //check function if user is global manager
    function tlive_useris_globalmanager($cm) {
        $context = context_module::instance($cm->id);
        return (has_capability('mod/tlive:isglobalmanager',  $context));
    }

    //check function if user has conference manager capabilities
    function tlive_useris_manager($cm,$id) {
        $context = context_module::instance($cm->id);
        return (has_capability('mod/tlive:conferencemanagercandidate',  $context , $id, false));
    }


   // $role = ( $tlive->confmanid == $USER->id || ($CFG->tlive_admincf && is_siteadmin()) || ($CFG->tlive_managercf && tlive_useris_globalmanager($cm) ) ) ? '2' : '1';
   
   //Retrieve multiple confmanid with explode from database and in_array check with $USER->id
   $role = ( in_array($USER->id, explode(',', $tlive->confmanid)) && tlive_useris_manager($cm,$USER->id) || ($CFG->tlive_admincf && is_siteadmin()) || ($CFG->tlive_managercf && tlive_useris_globalmanager($cm) ) ) ? '2' : '1';

    $lang = strtoupper($USER->lang);
    $xmlrequest = teleskill_beginrequest('3');

    $xmlrequest .= TLIVE_INDENT.'<clientcode>'.$CFG->tlive_clientcode.'</clientcode>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<roomid>'.$tlive->roomid.'</roomid>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<lmsuserid>'.$USER->id.'</lmsuserid>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<role>'.$role.'</role>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<name>'.fullname($USER).'</name>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<lang>'.$lang.'</lang>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<email>'.$USER->email.'</email>'.TLIVE_NEWLINE;
    $xmlrequest .= '</ews>';

    return tlive_getanswerfrom($xmlrequest);
}

/**
 * This function builds XML request to enable user access
 *
 * @param  object $tlive Tlive instance object
 * @return string  XML answer returned by Teleskill Live
 */
function tlive_enableuser_guest($tlive) {
    
    global $CFG;
    
    $role = '1';
    $lang = strtoupper($CFG->lang);
    $xmlrequest = teleskill_beginrequest('3');

    $xmlrequest .= TLIVE_INDENT.'<clientcode>'.$CFG->tlive_clientcode.'</clientcode>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<roomid>'.$tlive->roomid.'</roomid>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<lmsuserid>'.$tlive->guest_id.'</lmsuserid>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<role>'.$role.'</role>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<name>'.$tlive->firstname.' '. $tlive->lastname.'</name>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<lang>'.$lang.'</lang>'.TLIVE_NEWLINE;
    $xmlrequest .= '</ews>';

    return tlive_getanswerfrom($xmlrequest);
}

/**
 * This function checks if user has capability to be assigned as conference manager
 * @param object $cm Course module
 * @return boolean
 */
function tlive_useris_conferencemanagercandidate($cm) {
    $context = context_module::instance($cm->id);
    return (has_capability('mod/tlive:conferencemanagercandidate',  $context));
}

/**
 * This function checks if user has capability to manage tlive logs
 * @param object $cm Course module
 * @return boolean
 */
function tlive_usercan_managelogs($cm) {
    $context = context_module::instance( $cm->id);
    return (has_capability('mod/tlive:managelogs',  $context));
}

// Function checkroomrecs
function tlive_checkroomrecs($tlive, $force = false) {
	
	global $DB, $USER;
	
	$blnCheckRecs = true;
	
	// Verifica che sia passato un tempo minimo (minuti) per effettuare un check sulle rec della stanza	
	$minNextCheck = 60;
	if($room = $DB->get_record('tlive_rooms', array('roomid' => $tlive->roomid)) ) {		
		$lastCheck = $room->lastreccheck;
		$now = new DateTime("now", core_date::get_user_timezone_object());
		$now = $now->getTimeStamp();
		
		//echo ($now - $lastCheck).'<br>';
		//echo $minNextCheck*60;
		
		if(!$force && $lastCheck && ($now - $lastCheck)< $minNextCheck*60) {
			$blnCheckRecs = false;
		}
			
	}
	
	/*echo $blnCheckRecs;
	*/
	
	if($blnCheckRecs) {
		
		$xmlanswer = tlive_askforrecs($tlive);	
		$xmlstructure = simplexml_load_string($xmlanswer);	
	
		if ((int)$xmlstructure->errorcode == 0 && $xmlstructure->recs) {
			
			try {		
				
				$recs = $xmlstructure->recs;
			
				$transaction = $DB->start_delegated_transaction();
				
				if (!$DB->get_record('tlive_rooms', array('roomid' => $tlive->roomid))) {
			        $query="INSERT INTO {tlive_rooms} (roomid, synchronized) VALUES (?,?)";
			        $DB->execute($query, array($tlive->roomid, 0));
		        }
		
		        $recsfiles = array();    	
	     		
	
		        foreach ($recs->rec as $rec) {
		        	
		        	$recID = (int) $rec->attributes()->id;
				    $recTitle =  (string) $rec->attributes()->title;
				    $recStatus = (int)$rec->attributes()->status;
				    $recDate =  (string)$rec->attributes()->date;
				    $recDuration = (int) $rec->attributes()->duration; 	
				    
				    $daterec = new DateTime($recDate, new DateTimeZone("UTC"));
				    	
			    	$arrRec = array(		    	
	               		$recTitle,
	               		$tlive->roomid,    		 
	               		$recStatus,                  
	               		$daterec->getTimeStamp(),               		
	                	$recDuration,
	                	$recID
	                );  
				    
				    if (!$DB->get_record('tlive_recs', array('recid' =>$recID, 'roomid' => $tlive->roomid)) ) {	  	                       
	                    $query = 'INSERT INTO {tlive_recs} (title, roomid, status, timecreated, duration, recid) VALUES(?,?,?,?,?,?)';                    
				    } else {
				    	$query = 'UPDATE {tlive_recs} SET title=?, roomid=?, status=?, timecreated=?, duration=? WHERE recid=?';			    			    	
				    }			    
				    
				    $DB->execute($query, $arrRec);				      
		
		            if ($files = $rec->file) {
		
		                foreach ($files as $file) {
		                	
		                	$fileID = (int) $file->attributes()->id;			    	
					    	$fileFormat = (int) $file->attributes()->format;			    	
					    	$fileUrl = (string) $file->attributes()->src;
					    	$blnScorm = (int) $file->attributes()->scorm;
					    	$blnZip = (int) $file->attributes()->zip;                   
		                    
		                    $arrFile = array(                       
	                        	$recID,
	                       		$fileFormat,
	                       		$fileUrl,
	                       		$blnScorm,
	                        	$blnZip,
	                        	$fileID 
	                        );
		                    
		                   
		                    /* verify if this rec is not already saved into repository */
		                    if (!$result = $DB->get_record('tlive_recs_files', array('fileid' => $fileID)) ) {
		                    	$query = 'INSERT INTO {tlive_recs_files} (recid, format, url, scorm, zip, fileid) VALUES(?,?,?,?,?,?)';	                    	             	
		                    } else {
		                    	$query = 'UPDATE {tlive_recs_files} SET recid=?, format=?, url=?, scorm=?, zip=? WHERE fileid=?';	
		                    }      
		                    
		                    
		                    
		                    $DB->execute($query, $arrFile);	  
		
		                }
		            }
		            
		        }
		        
		        $now = new DateTime("now", core_date::get_user_timezone_object());
		        
		        $query = "UPDATE {tlive_rooms} SET lastreccheck=? WHERE roomid=?";
		        $DB->execute($query, array($now->getTimeStamp(), $tlive->roomid));	       
		        
		    	$transaction->allow_commit();
			
			} catch(Exception $e) {
			     $transaction->rollback($e);
			}
	    }
	
	}
		
}

// function askforrecs
function tlive_askforrecs($tlive) {
	
	global $CFG;

    $xmlrequest = teleskill_beginrequest('20');

    $xmlrequest .= TLIVE_INDENT.'<clientcode>'.$CFG->tlive_clientcode.'</clientcode>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<roomid>'.$tlive->roomid.'</roomid>'.TLIVE_NEWLINE;   
    $xmlrequest .= '</ews>';

    return tlive_getanswerfrom($xmlrequest);

}

/**
 * This function prints an html table that summarizes channel data
 *
 * @param string $xmlanswer XML answer from Teleskill Live server
 */
function tlive_displaychanneldata($xmlanswer) {

    $credits = tlive_readxmlanswer($xmlanswer, 'credits');
    $users = tlive_readxmlanswer($xmlanswer,  'users');
    $rooms = tlive_readxmlanswer($xmlanswer,  'rooms');

    $strcredits = ($credits) ? get_string('credits', 'tlive', $credits) : get_string('flatchannel', 'tlive');
    $strusers   = ($users)   ? $users : get_string('unlimited', 'tlive');
    $strrooms   = $rooms     ? get_string('rooms', 'tlive', $rooms) : get_string('unlimitedrooms', 'tlive');

    $table = new html_table();
    $table->head = array(get_string('header_credits', 'tlive'),
                         get_string('header_users', 'tlive'),
                         get_string('header_rooms', 'tlive'));
    $table->data[] = array($strcredits,  $strusers,  $strrooms);
    $table->align = array('left', 'left', 'left');
    $table->wrap = array(false, false, false);
    $table->size = array('33%', '33%', '34%');

    echo html_writer::table($table);
    print_string('channelloginfo',  'tlive');
}


/**
 * This function builds XML request to get room's logs
 *
 * @param  int $complete If 1 ask complete log, 0 otherwise
 * @param  object $tlive Tlive instance object
 * @return string  XML answer returned by Teleskill Live
 */
function tlive_askforlogs($complete,  $tlive) {
    global $CFG;

    $xmlrequest = teleskill_beginrequest('16');

    $xmlrequest .= TLIVE_INDENT.'<clientcode>'.$CFG->tlive_clientcode.'</clientcode>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<roomid>'.$tlive->roomid.'</roomid>'.TLIVE_NEWLINE;
    $xmlrequest .= TLIVE_INDENT.'<complete>'.$complete.'</complete>'.TLIVE_NEWLINE;
    $xmlrequest .= '</ews>';

    return tlive_getanswerfrom($xmlrequest);
}


/**
 * This function saves logs data to db from XML log answer
 *
 * @param string $xmlanswer XML answer returned by Teleskill Live
 */
function tlive_savelogs($xmlanswer) {

    global $DB;

    $xmlstructure = simplexml_load_string($xmlanswer);
    /* Statistics are passed by Tlive in Europe/Rome local time*/
    date_default_timezone_set('Europe/Rome');

    if ($xmlstructure->Room) {

        $room = $xmlstructure->Room;
        $roomid = (int) $xmlstructure->Room->attributes()->ID;

        if (!$DB->get_record('tlive_rooms', array('roomid' => $roomid))) {

            $query="INSERT INTO {tlive_rooms} (roomid, synchronized) VALUES (?, 0)";
            $DB->execute($query, array($roomid));
        }

        foreach ($room->RoomSession as $roomsession) {

            $roomsessionid = (int)$roomsession->attributes()->ID;
            $startdate = strtotime($roomsession->attributes()->StartDate);
            $enddate = strtotime($roomsession->attributes()->EndDate);

            if (!$DB->get_record('tlive_rooms_sessions', array('roomsessionid' => $roomsessionid))) {
                $query="INSERT INTO {tlive_rooms_sessions} VALUES (?, ?, ?, ?, ?)";

                /* Notice: you can't use insert_record because primary key (roomid) is already defined,
                 * use execute SQL */
                $DB->execute($query, array($roomsessionid, $roomid, $startdate, $enddate, 0));
            }

            /* if polls sessions exist, save them */
            if ($roomsession->Polls) {
                tlive_getpolls($roomsession->Polls, $roomsessionid);
            }

            /* if users sessions exist, save them */
            if ($roomsession->UserSessions) {
                tlive_getusersessions($roomsession->UserSessions, $roomsessionid);
            }

            /* if lessions sessions exist, save them */
            if ($roomsession->LessonSessions) {
                tlive_getlessonsessions($roomsession->LessonSessions, $roomsessionid);
            }

        }

    }

}


/**
 * This function saves polls data to db, from polls simplexml object
 *
 * @param object $polls Polls data simplexml object
 * @param int $roomsessionid Room session id
 */
function tlive_getpolls($polls, $roomsessionid) {

    global $DB;

    foreach ($polls->Poll as $poll) {

        $pollid = (int)$poll->attributes()->ID;
        /* replace "?" and prevent any quotes */
        //$question = addslashes(str_replace("?",  '',  $poll->attributes()->Question));
        $question = (string)$poll->attributes()->Question;

        if (!$DB->get_record('tlive_polls', array('pollid' => $pollid))) {
            $query="INSERT INTO {tlive_polls} VALUES (?, ?, ?)";
            $DB->execute($query, array($pollid, $question, $roomsessionid));
        }

        if ($poll->Answers) {
            tlive_getanswers($poll->Answers, $pollid);
        }

        if ($poll->PollSessions) {
            tlive_getpollsessions($poll->PollSessions, $pollid);
        }

    }

}


/**
 * This function saves poll answers data to db, from answers simplexml object
 *
 * @param object $answers Answers simplexml object
 * @param int $pollid Poll id
 */
function tlive_getanswers($answers, $pollid) {

    global $DB;

    foreach ($answers->Answer as $answer) {

        $answerid = (int)$answer->attributes()->ID;
        $sequence = (int)$answer->attributes()->Sequence;

        //$answerstr = addslashes(str_replace("?",  '', $answer->attributes()->Answer));
        $answerstr = (string)$answer->attributes()->Answer;
        $correct = (int)$answer->attributes()->Correct;

        if (!$DB->get_record('tlive_polls_answers', array('answerid' => $answerid))) {
            $query="INSERT INTO {tlive_polls_answers} VALUES (?, ?, ?, ?, ?)";
            $DB->execute($query, array($answerid, $pollid, $answerstr, $sequence, $correct));
        }

    }

}


/**
 * This function saves poll sessions data to db, from answers simplexml object
 *
 * @param object $pollsessions Answers simplexml object
 * @param int $pollid Poll id
 */
function tlive_getpollsessions($pollsessions, $pollid) {

    global $DB;

    foreach ($pollsessions->PollSession as $pollsession) {

        $pollsessionid = (int)$pollsession->attributes()->ID;
        $opendate = (int)strtotime($pollsession->attributes()->OpenDate);
        $closedate = (int)strtotime($pollsession->attributes()->CloseDate);        
        

        if (!$DB->get_record('tlive_polls_sessions', array('pollsessionid' => $pollsessionid))) {
            $query="INSERT INTO {tlive_polls_sessions} VALUES (?, ?, ?, ?)";
            $DB->execute($query, array($pollsessionid, $pollid, $opendate, $closedate));
        }

        $useranswers = $pollsession->UserAnswer;
        tlive_getuseranswers($useranswers, $pollsessionid);
    }
}


/**
 * This function saves user answers data to db, from user answers simplexml object
 *
 * @param object $useranswers Answers simplexml object
 * @param int $pollsessionid Poll session id
 */
function tlive_getuseranswers($useranswers, $pollsessionid) {

    global $DB;

    foreach ($useranswers as $useranswer) {

        $useranswerid = (int)$useranswer->attributes()->ID;
        $usersessionid = (int)$useranswer->attributes()->UserSessionID;
        $answerid = (int)$useranswer->attributes()->AnswerID;
        $creationdate = strtotime($useranswer->attributes()->CreationDate);

        if (!$DB->get_record('tlive_users_poll_answers', array('userspollanswerid' => $useranswerid))) {
            $query="INSERT INTO {tlive_users_poll_answers}
                    VALUES (?, ?, ?, ?, ?)";
            $DB->execute($query, array($useranswerid, $pollsessionid, $usersessionid, $answerid, $creationdate));
        }

    }
}

/**
 * This function saves user sessions data to db, from user sessions simplexml object
 *
 * @param object $usersessions Answers simplexml object
 * @param int $roomsessionid Room session id
 */
function tlive_getusersessions($usersessions, $roomsessionid) {

    global $DB;

    foreach ($usersessions->UserSession as $usersession) {

        $usersessionid = (int) $usersession->attributes()->ID;       
        
        if(preg_match("/^guest_\d+$/", $usersession->attributes()->UserID)) {
        	$userid = 0;
        	$guest_user_id = (int) str_replace('guest_','',$usersession->attributes()->UserID);
        	
        } else {
        	$userid = (int) $usersession->attributes()->UserID;
        	$guest_user_id = 0;
        }
        
        
        $roleid = (int) $usersession->attributes()->RoleID;
        $startdate = strtotime($usersession->attributes()->StartDate);
        $enddate = strtotime($usersession->attributes()->EndDate);
        /* non presente nel xml
        * $ipaddress = $usersession->attributes()->IpAddress; */

        if (!$DB->get_record('tlive_user_sessions', array('usersessionid' => $usersessionid))) {
            $query="INSERT INTO {tlive_user_sessions} (usersessionid, iduser, roomsessionid, roleid, startdate, enddate, ipaddress, idguest)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
            $DB->execute($query, array($usersessionid, $userid, $roomsessionid, $roleid, $startdate, $enddate, '', $guest_user_id));
        }
    }
}

/**
 * This function saves lesson sessions data to db, from lesson sessions simplexml object
 *
 * @param object $lessonsessions Lesson sessions simplexml object
 * @param int $roomsessionid Room session id
 */
function tlive_getlessonsessions($lessonsessions, $roomsessionid) {

    global $DB;

    foreach ($lessonsessions->LessonSession as $lessonsession) {

        $lessonsessionid = (int) $lessonsession->attributes()->ID;
        $lessonstartdate = strtotime($lessonsession->attributes()->StartDate);
        $lessonenddate = strtotime($lessonsession->attributes()->EndDate);
        $lessonduration = (int) $lessonsession->attributes()->Duration;

        if (!$DB->get_record('tlive_lesson_sessions', array('lessonsessionid' => $lessonsessionid))) {
            $query="INSERT INTO {tlive_lesson_sessions}
                    VALUES (?, ?, ?, ?, ?)";
            $DB->execute($query, array($lessonsessionid, $roomsessionid, $lessonstartdate, $lessonenddate, $lessonduration));
        }
        tlive_setuserlessonsession($lessonsession, $roomsessionid);
    }
}


/**
 * This function compute the effective user lesson session, crossing user sessions data with lesson sessions data
 * and save it to db
 *
 * @param object $lessonsession Lesson session simplexml object
 * @param int $roomsessionid Room session id
 */
function tlive_setuserlessonsession($lessonsession, $roomsessionid) {

    global $DB;

    $lessonstartdate = strtotime($lessonsession->attributes()->StartDate);
    $lessonenddate   = strtotime($lessonsession->attributes()->EndDate);
    $lessonsessionid = (int)$lessonsession->attributes()->ID;

    /* extract all users sessions */
    if ($usersessions = $DB->get_records('tlive_user_sessions', array('roomsessionid' => $roomsessionid))) {

        /* for each user session check if it intersects lesson session, if true, save it to db */
        foreach ($usersessions as $usersession) {

            $cond1 = ($lessonstartdate>= $usersession->startdate && $lessonstartdate<=$usersession->enddate);
            $cond2 = ($lessonenddate>= $usersession->startdate && $lessonenddate<=$usersession->enddate);
            $cond3 = ($lessonstartdate<= $usersession->startdate && $lessonenddate>=$usersession->startdate);

            if ( $cond1 || $cond2 || $cond3 ) {

                /* save user lesson session  */
                $userlessonsessionstart = max($lessonstartdate, $usersession->startdate);
                $userlessonsessionend = min($lessonenddate, $usersession->enddate);
                $usersessionid = $usersession->usersessionid;
                $duration = $userlessonsessionend - $userlessonsessionstart;

                $check = $DB->get_record('tlive_lesson_user_sessions',
                                          array('lessonsessionid' => $lessonsessionid, 'usersessionid' => $usersessionid));
                if (!$check) {
                    $query ="INSERT INTO {tlive_lesson_user_sessions}
                             VALUES (NULL, ?, ?, ?, ?, ?)";
                    $DB->execute($query, array($lessonsessionid, $usersessionid, $userlessonsessionstart, $userlessonsessionend, $duration));
                }
            }
        }
    }
}

/**
 * This function is called by events.php and format user timestamp to ISO 8601
 * @param int $date timestamp format
 * @return string $aux date ISO8601
 */
function tlive_get_isodatetime($date) {
	
	
    $year = userdate($date, '%Y');
    $month = userdate($date, '%m');
    $day = str_pad(userdate($date, '%d'), 2, '0', STR_PAD_LEFT);
    $hour = userdate($date, '%H');
    $minutes = userdate($date, '%M');
    $seconds = userdate($date, '%S');
    
    $dtaux = new DateTime();
    $dtaux->setTimeStamp($date);
	/* convert to system time zone */
	$dtaux->setTimeZone(new DateTimeZone(get_user_timezone()) );

    $usertimeoffset = $dtaux->getOffset();    

    $usertimeoffset = $usertimeoffset/3600;   

    /* compute daylight saving offset */
    $dst = dst_offset_on($date)/3600;
   
    
    $usertimeoffset = $usertimeoffset + $dst;

    $usertimeoffset = str_pad($usertimeoffset, 2, 0, STR_PAD_LEFT);

    /* trim day used because userdate() function inserts a space before 2-digit days (moodle bug) MDL-11601*/
    $aux = $year.'-'.$month.'-'.trim($day).'T'.$hour.':'.$minutes.':'.$seconds.'+'.$usertimeoffset.':00';

    return $aux;

}

/**
 * This function print an html table that summarizes poll data
 *
 * @param int $pollsessionid Poll session id
 * @return object $table Html object table
 */
function tlive_buildtablepolls($pollsessionid) {

    global $DB,$CFG,$COURSE,$cm;

    //check if codicefiscale short code exist in admin settings page
    if(!empty($CFG->tlive_codicefiscalecode)){
        
        $codicefiscalecode = $CFG->tlive_codicefiscalecode;

        $query = "SELECT ui.data as codicefiscale
        FROM {user_info_data} AS ui    
        JOIN {user_info_field} as uif ON ui.fieldid = uif.id AND uif.shortname = '$codicefiscalecode' "; 
        
        //check short code on database
        if (!$DB->get_records_sql($query)) {
            $codicefiscalecode = '';
        }
        
    }else{
        $codicefiscalecode = '';
    }

    $strfdate = get_string('displaydateformat',  'tlive');
    $strftime = get_string('displaytimeformat',  'tlive');
    $query = "SELECT p.question, ps.opendate, ps.closedate
               FROM {tlive_polls_sessions} as ps
               INNER JOIN {tlive_polls} as p ON ps.pollid=p.pollid
               WHERE ps.pollsessionid='$pollsessionid'";
    $poll = $DB->get_record_sql($query);
    $title = get_string('question', 'tlive').": $poll->question,  ".
             get_string('poll_session', 'tlive').' '.userdate($poll->opendate, $strfdate.' '.$strftime).
             " - ".userdate($poll->closedate, $strftime);
    $tlive_roles = array(
    	1 => get_string('participant', 'tlive'),
    	2 => get_string('conferencemanager', 'tlive'),
    	3 => get_string('speaker', 'tlive')
    );

    /* sort fields */
    $sortparams = tlive_sort_tablebyfield();
    $qrstr = $sortparams->qrStr;
    $sortfield = $sortparams->sort_field;
    $sortdirection = $sortparams->sort_direction;
    $newsortdirection = $sortparams->new_sort_direction;

    /* table heading */
    $headinguser = "<a href='$qrstr&sort=firstname&order=$newsortdirection'>".get_string('user', 'tlive')."</a>";
    $headingrole = "<a href='$qrstr&sort=roleid&order=$newsortdirection'>".get_string('role', 'tlive')."</a>";
    //check if codicefiscale filed exist then insert codicefiscale field in heading
    if(!empty($codicefiscalecode)){
        $headingcodicefiscale = "<a href='$qrstr&sort=roleid&order=$newsortdirection'>".get_string('tlive_codicefiscale', 'tlive')."</a>";
    }
    $headingcourse = "<a href='$qrstr&sort=roleid&order=$newsortdirection'>".get_string('course_name', 'tlive')."</a>";
    $headingactivity = "<a href='$qrstr&sort=roleid&order=$newsortdirection'>".get_string('activity', 'tlive')."</a>";
    $headinganswer = "<a href='$qrstr&sort=answer&order=$newsortdirection'>".get_string('answer', 'tlive')."</a>";
    $headingcorrect = "<a href='$qrstr&sort=correct&order=$newsortdirection'>".get_string('correct', 'tlive')."</a>";

    $table = new html_table();
    //check if codicefiscale filed exist
    if(!empty($codicefiscalecode)){
        $table->head  = array($headinguser, $headingrole, $headingcodicefiscale, $headingcourse, $headingactivity, $headinganswer, $headingcorrect);
    }else{    
        $table->head  = array($headinguser, $headingrole, $headingcourse, $headingactivity, $headinganswer, $headingcorrect);
    }
    $table->data = array();
    /* title is a custom propriety, not present in html_table class */
    $table->title = $title;
    //Check if codicefiscale shortcode field is not empty
    if(!empty($codicefiscalecode)){

        $query = "SELECT upa.userspollanswerid, u.firstname, u.lastname, u.email, ps.correct, us.roleid,
        ps.answer, us.startdate, us.enddate, us.iduser ,ui.data as codicefiscale
        FROM ({tlive_users_poll_answers} AS upa
        INNER JOIN {tlive_user_sessions} AS us ON upa.usersessionid=us.usersessionid
        INNER JOIN {tlive_polls_answers} AS ps ON upa.answerid=ps.answerid)
        INNER JOIN {user} as u ON us.iduser = u.id
        INNER JOIN {user_info_data} AS ui ON us.iduser=ui.userid
        JOIN {user_info_field} as uif ON ui.fieldid = uif.id AND uif.shortname = '$codicefiscalecode' 
        WHERE pollsessionid='$pollsessionid' AND us.idguest=0
        UNION 
        SELECT upa.userspollanswerid, gu.firstname, gu.lastname,gu.email, ps.correct, us.roleid,
        ps.answer, us.startdate, us.enddate, us.iduser ,ui.data as codicefiscale
        FROM ({tlive_users_poll_answers} AS upa
        INNER JOIN {tlive_user_sessions} AS us ON upa.usersessionid=us.usersessionid
        INNER JOIN {tlive_polls_answers} AS ps ON upa.answerid=ps.answerid)
        INNER JOIN {tlive_guest_users} AS gu ON gu.id=us.idguest
        INNER JOIN {user_info_data} AS ui ON us.iduser=ui.userid
        JOIN {user_info_field} as uif ON ui.fieldid = uif.id AND uif.shortname = '$codicefiscalecode'
        WHERE pollsessionid='$pollsessionid' AND us.idguest<>0	            
        ORDER BY $sortfield $sortdirection";

    }else{

        //codicefiscale shortcode field is empty
        $query = "SELECT upa.userspollanswerid, u.firstname, u.lastname, u.email, ps.correct, us.roleid,
        ps.answer, us.startdate, us.enddate, us.iduser
        FROM ({tlive_users_poll_answers} AS upa
        INNER JOIN {tlive_user_sessions} AS us ON upa.usersessionid=us.usersessionid
        INNER JOIN {tlive_polls_answers} AS ps ON upa.answerid=ps.answerid)
        INNER JOIN {user} as u ON us.iduser = u.id
        WHERE pollsessionid='$pollsessionid' AND us.idguest=0
        UNION 
        SELECT upa.userspollanswerid, gu.firstname, gu.lastname,gu.email, ps.correct, us.roleid,
        ps.answer, us.startdate, us.enddate, us.iduser
        FROM ({tlive_users_poll_answers} AS upa
        INNER JOIN {tlive_user_sessions} AS us ON upa.usersessionid=us.usersessionid
        INNER JOIN {tlive_polls_answers} AS ps ON upa.answerid=ps.answerid)
        INNER JOIN {tlive_guest_users} AS gu ON gu.id=us.idguest
        WHERE pollsessionid='$pollsessionid' AND us.idguest<>0	            
        ORDER BY $sortfield $sortdirection";
    }                

    if ($result = $DB->get_records_sql($query)) {

        foreach ($result as $r) {
        	
        	$usertext = $r->firstname.' '.$r->lastname;
        	$roletext = $tlive_roles[$r->roleid];
        	
        	if ($r->iduser == 0) {
        		$usertext .= !empty($r->email) ? ' ('.$r->email.')' : '';
            	$roletext = $roletext .' ('.get_string("guest").')';	
            }

            if ($r->correct==1) {
                $correct = get_string('answer_correct', 'tlive');
            } else {
                $correct = get_string('answer_wrong', 'tlive');
            }

            //check if codice fiscale exist then push codicefiscale
            if(!empty($codicefiscalecode)){
                $row = array($usertext, $roletext, $r->codicefiscale, $COURSE->fullname, $cm->name, $r->answer, $correct);
            }else{
                $row = array($usertext, $roletext, $COURSE->fullname, $cm->name, $r->answer, $correct);
            }

            array_push($table->data, $row);

        }
    }

    return $table;

}

/**
 * This function sort html table data by a field parsed through GET object
 *
 * @return object $sortparams Sort params
 */
function tlive_sort_tablebyfield() {

    $qrstr = $_SERVER['PHP_SELF']."?".$_SERVER['QUERY_STRING'];

    $sort = optional_param('sort', '', PARAM_TEXT);
    $order = optional_param('order', '', PARAM_TEXT);

    if (empty($sort) && empty($order)) {

        $sortfield = "startdate, enddate ";
        $sortdirection ="ASC";

    } else {
        /* clear old sort and order rule */
        $qrstr = str_replace('&sort='.$sort.'&order='.$order, '', $qrstr);

        $sortfield = $sort;
        $sortdirection = $order;
    }

    $newsortdirection = $sortdirection;
    $newsortdirection = $newsortdirection == 'ASC'? 'DESC':'ASC';

    $sortparams = new stdClass();
    /* querystring */
    $sortparams->qrStr = $qrstr;
    /* sort field */
    $sortparams->sort_field = $sortfield;
    /* actual sort direction */
    $sortparams->sort_direction = $sortdirection;
    /* next sort direction (after refresh) */
    $sortparams->new_sort_direction = $newsortdirection;

    return $sortparams;

}


/**
 * This function print an html table that summarizes room access data
 *
 * @param int $roomsessionid Room session id
 * @param int $lessonsessionid Lesson session id
 * @param int $total session chk
 * @return object $table Html object table
 */
function tlive_buildtableaccess($roomsessionid, $lessonsessionid=null , $total=null) {

    global $DB,$CFG,$COURSE,$cm;
    
    //check if codicefiscale short code exist in admin settings page
    if(!empty($CFG->tlive_codicefiscalecode)){
        
        $codicefiscalecode = $CFG->tlive_codicefiscalecode;

        $query = "SELECT ui.data as codicefiscale
        FROM {user_info_data} AS ui    
        JOIN {user_info_field} as uif ON ui.fieldid = uif.id AND uif.shortname = '$codicefiscalecode' "; 
        
        //check short code on database
        if (!$DB->get_records_sql($query)) {
            $codicefiscalecode = '';
        }
        
    }else{
        $codicefiscalecode = '';
    }

    $rsid = optional_param('rsid', 0, PARAM_INT);
    $lsid = optional_param('lsid', 0, PARAM_INT);

    $strfdate = get_string('displaydateformat',  'tlive');
    $strftime = get_string('displaytimeformat',  'tlive');
    
    $tlive_roles = array(
    	1 => get_string('participant', 'tlive'),
    	2 => get_string('conferencemanager', 'tlive'),
    	3 => get_string('speaker', 'tlive')
    );

    /* Print title */
    if ($rsid != 0 && $lsid != 0) {

        $lessonsessionrecord = $DB->get_record('tlive_lesson_sessions', array('lessonsessionid' => $lsid));
        $title = get_string('access_list', 'tlive')." - ".
        get_string("room_session",  "tlive").userdate($lessonsessionrecord->startdate, $strfdate).' '.get_string("from", "tlive").
        userdate($lessonsessionrecord->startdate, $strftime).' '.get_string("to", "tlive").
        userdate($lessonsessionrecord->enddate, $strftime);

    } else if ($rsid != 0) {

        $roomsessionrecord = $DB->get_record('tlive_rooms_sessions', array('roomsessionid' => $rsid));
        $title = get_string('access_list', 'tlive')." - ".
        get_string("room_session",  "tlive").userdate($roomsessionrecord->startdate, $strfdate).' '.get_string("from", "tlive").
        userdate($roomsessionrecord->startdate, $strftime).' '.
        get_string("to", "tlive").userdate($roomsessionrecord->enddate, $strftime);

    } else {
        $title="";
    }

    /* Sort fields */
    $sortparams = tlive_sort_tablebyfield();
    $qrstr = $sortparams->qrStr;
    $sortfield = $sortparams->sort_field;
    $sortdirection = $sortparams->sort_direction;
    $newsortdirection = $sortparams->new_sort_direction;

    /* table heading */
    $headinguser = "<a href='$qrstr&sort=firstname&order=$newsortdirection'>".get_string('user', 'tlive')."</a>";
    $headingrole = "<a href='$qrstr&sort=roleid&order=$newsortdirection'>".get_string('role', 'tlive')."</a>";
    //check if codicefiscale filed exist then insert codicefiscale field in heading
    if(!empty($codicefiscalecode)){
        $headingcodicefiscale = "<a href='$qrstr&sort=roleid&order=$newsortdirection'>".get_string('tlive_codicefiscale', 'tlive')."</a>";
    }
    $headingcourse = "<a href='$qrstr&sort=roleid&order=$newsortdirection'>".get_string('course_name', 'tlive')."</a>";
    $headingactivity = "<a href='$qrstr&sort=roleid&order=$newsortdirection'>".get_string('activity', 'tlive')."</a>";
    
        $headingstart = "<a href='$qrstr&sort=startdate&order=$newsortdirection'>".get_string('start', 'tlive')."</a>";
        $headingend = "<a href='$qrstr&sort=enddate&order=$newsortdirection'>".get_string('end', 'tlive')."</a>";
    
    //check if lessionsessionid is set then assign duration in table heading
    if(isset($lessonsessionid)){
        $headingduration = "<a href='$qrstr&sort=roleid&order=$newsortdirection'>".get_string('header_duration', 'tlive')."</a>";
    }
    /* headingip sort=ipaddress */

    $tableuserssessions = new html_table();

    //check if codicefiscale filed exist
    if(!empty($codicefiscalecode)){

        //check if lesson id is set then push to table head
        if(isset($lessonsessionid)){

            //check lesson total duration
            if(isset($total) && $total == 'total'){

                $tableuserssessions->head  = array($headinguser, $headingrole, $headingcodicefiscale, $headingcourse, $headingactivity, $headingduration);

            }else{

                $tableuserssessions->head  = array($headinguser, $headingrole, $headingcodicefiscale, $headingcourse, $headingactivity, $headingstart, $headingend , $headingduration);

            }

        }else{

            $tableuserssessions->head  = array($headinguser, $headingrole, $headingcodicefiscale, $headingcourse, $headingactivity, $headingstart, $headingend);
        }

    }else{

        //check if lesson id is set then push to table head
        if(isset($lessonsessionid)){

            //check lesson total duration
            if(isset($total) && $total == 'total'){

                $tableuserssessions->head  = array($headinguser, $headingrole, $headingcourse, $headingactivity, $headingduration);

            }else{

                $tableuserssessions->head  = array($headinguser, $headingrole, $headingcourse, $headingactivity, $headingstart, $headingend, $headingduration);
            } 

        }else{

            $tableuserssessions->head  = array($headinguser, $headingrole, $headingcourse, $headingactivity, $headingstart, $headingend);

        }

    }

    $tableuserssessions->data = array();
    $tableuserssessions->title = $title;

    //Check if codicefiscale shortcode field is not empty
    if(!empty($codicefiscalecode)){

        /* Flag to call print user lession session task */
        if (isset($lessonsessionid)) {
            
            //If total is set then show total user lesson session duration
            if(isset($total) && $total == 'total'){

                $query = "SELECT uls.sessionid, u.firstname, u.lastname, u.email, uls.startdate, uls.enddate, us.roleid, us.ipaddress, us.iduser , ui.data as codicefiscale , sum(uls.duration) as duration
                FROM {tlive_lesson_user_sessions} AS uls
                INNER JOIN {tlive_user_sessions} AS us ON uls.usersessionid=us.usersessionid
                INNER JOIN {user} AS u ON us.iduser=u.id
                INNER JOIN {user_info_data} AS ui ON us.iduser=ui.userid
                JOIN {user_info_field} as uif ON ui.fieldid = uif.id AND uif.shortname = '$codicefiscalecode'
                WHERE uls.lessonsessionid='$lessonsessionid' AND us.idguest=0
                GROUP BY u.email
                UNION
                SELECT  uls.sessionid, gu.firstname, gu.lastname, gu.email, uls.startdate, uls.enddate, us.roleid, us.ipaddress, us.iduser , ui.data as codicefiscale , sum(uls.duration) as duration
                FROM {tlive_lesson_user_sessions} AS uls
                INNER JOIN {tlive_user_sessions} AS us ON uls.usersessionid=us.usersessionid
                INNER JOIN {tlive_guest_users} AS gu ON gu.id=us.idguest
                INNER JOIN {user_info_data} AS ui ON us.iduser=ui.userid
                JOIN {user_info_field} as uif ON ui.fieldid = uif.id AND uif.shortname = '$codicefiscalecode'
                WHERE uls.lessonsessionid='$lessonsessionid' AND us.idguest<>0  
                GROUP BY gu.email  
                ORDER BY $sortfield $sortdirection";

            } else{

                $query = "SELECT uls.sessionid, u.firstname, u.lastname, u.email, uls.startdate, uls.enddate, us.roleid, us.ipaddress, us.iduser , ui.data as codicefiscale , uls.duration
                FROM {tlive_lesson_user_sessions} AS uls
                INNER JOIN {tlive_user_sessions} AS us ON uls.usersessionid=us.usersessionid
                INNER JOIN {user} AS u ON us.iduser=u.id
                INNER JOIN {user_info_data} AS ui ON us.iduser=ui.userid
                JOIN {user_info_field} as uif ON ui.fieldid = uif.id AND uif.shortname = '$codicefiscalecode'
                WHERE uls.lessonsessionid='$lessonsessionid' AND us.idguest=0
                UNION
                SELECT  uls.sessionid, gu.firstname, gu.lastname, gu.email, uls.startdate, uls.enddate, us.roleid, us.ipaddress, us.iduser , ui.data as codicefiscale ,uls.duration
                FROM {tlive_lesson_user_sessions} AS uls
                INNER JOIN {tlive_user_sessions} AS us ON uls.usersessionid=us.usersessionid
                INNER JOIN {tlive_guest_users} AS gu ON gu.id=us.idguest
                INNER JOIN {user_info_data} AS ui ON us.iduser=ui.userid
                JOIN {user_info_field} as uif ON ui.fieldid = uif.id AND uif.shortname = '$codicefiscalecode'
                WHERE uls.lessonsessionid='$lessonsessionid' AND us.idguest<>0      
                ORDER BY $sortfield $sortdirection";

            }   

        } else {

            $query = "SELECT us.usersessionid, u.firstname, u.lastname, u.email, us.roleid, us.startdate, us.enddate, us.iduser , ui.data as codicefiscale  FROM {tlive_user_sessions} AS us
            INNER JOIN {user} AS u ON us.iduser=u.id
            INNER JOIN {user_info_data} AS ui ON us.iduser=ui.userid
            JOIN {user_info_field} as uif ON ui.fieldid = uif.id AND uif.shortname = '$codicefiscalecode'        
            WHERE us.roomsessionid='$roomsessionid' AND us.idguest=0
            UNION                 
            SELECT us.usersessionid, gu.firstname, gu.lastname, gu.email, us.roleid, us.startdate, us.enddate, us.iduser , ui.data as codicefiscale  FROM {tlive_user_sessions} AS us
            INNER JOIN {tlive_guest_users} AS gu ON gu.id=us.idguest
            INNER JOIN {user_info_data} AS ui ON us.iduser=ui.userid
            JOIN {user_info_field} as uif ON ui.fieldid = uif.id AND uif.shortname = '$codicefiscalecode'            
            WHERE us.roomsessionid='$roomsessionid' AND us.idguest<>0
            ORDER BY $sortfield $sortdirection";                 
        }
    } 
    
    //codicefiscale shortcode field is empty
    else{
         /* Flag to call print user lession session task */
        if (isset($lessonsessionid)) {

            // build user lession session total duration
            if(isset($total) && $total == 'total'){

                $query = "SELECT uls.sessionid, u.firstname, u.lastname, u.email, uls.startdate, uls.enddate, us.roleid, us.ipaddress, us.iduser , sum(uls.duration) as duration
                FROM {tlive_lesson_user_sessions} AS uls
                INNER JOIN {tlive_user_sessions} AS us ON uls.usersessionid=us.usersessionid
                INNER JOIN {user} AS u ON us.iduser=u.id
                WHERE uls.lessonsessionid='$lessonsessionid' AND us.idguest=0
                GROUP BY u.email
                UNION
                SELECT  uls.sessionid, gu.firstname, gu.lastname, gu.email, uls.startdate, uls.enddate, us.roleid, us.ipaddress, us.iduser, sum(uls.duration) as duration
                FROM {tlive_lesson_user_sessions} AS uls
                INNER JOIN {tlive_user_sessions} AS us ON uls.usersessionid=us.usersessionid
                INNER JOIN {tlive_guest_users} AS gu ON gu.id=us.idguest
                WHERE uls.lessonsessionid='$lessonsessionid' AND us.idguest<>0
                GROUP BY gu.email       
                ORDER BY $sortfield $sortdirection";

            }else{

                $query = "SELECT uls.sessionid, u.firstname, u.lastname, u.email, uls.startdate, uls.enddate, us.roleid, us.ipaddress, us.iduser , uls.duration
                FROM {tlive_lesson_user_sessions} AS uls
                INNER JOIN {tlive_user_sessions} AS us ON uls.usersessionid=us.usersessionid
                INNER JOIN {user} AS u ON us.iduser=u.id
                WHERE uls.lessonsessionid='$lessonsessionid' AND us.idguest=0
                UNION
                SELECT  uls.sessionid, gu.firstname, gu.lastname, gu.email, uls.startdate, uls.enddate, us.roleid, us.ipaddress, us.iduser, uls.duration
                FROM {tlive_lesson_user_sessions} AS uls
                INNER JOIN {tlive_user_sessions} AS us ON uls.usersessionid=us.usersessionid
                INNER JOIN {tlive_guest_users} AS gu ON gu.id=us.idguest
                WHERE uls.lessonsessionid='$lessonsessionid' AND us.idguest<>0      
                ORDER BY $sortfield $sortdirection";

            }

        } else {

            $query = "SELECT us.usersessionid, u.firstname, u.lastname, u.email, us.roleid, us.startdate, us.enddate, us.iduser  FROM {tlive_user_sessions} AS us
            INNER JOIN {user} AS u ON us.iduser=u.id       
            WHERE us.roomsessionid='$roomsessionid' AND us.idguest=0
            UNION                 
            SELECT us.usersessionid, gu.firstname, gu.lastname, gu.email, us.roleid, us.startdate, us.enddate, us.iduser  FROM {tlive_user_sessions} AS us
            INNER JOIN {tlive_guest_users} AS gu ON gu.id=us.idguest   
            WHERE us.roomsessionid='$roomsessionid' AND us.idguest<>0
            ORDER BY $sortfield $sortdirection";                 
        }
    }    

    if (!$usersessions = $DB->get_records_sql($query)) {
        echo get_string('nousersessionavailable', 'tlive');
    } else {

        foreach ($usersessions as $u) {        

            $usertext = $u->firstname.' '.$u->lastname;
            $roletext = $tlive_roles[$u->roleid];
            if ($u->iduser == 0) {
            	
            	$usertext .= !empty($u->email) ? ' ('.$u->email.')' : '';           	
            	
            	$roletext = $roletext .' ('.get_string("guest").')';	
            }
            $start = userdate($u->startdate, $strfdate.' '.$strftime);
            $end = userdate($u->enddate, $strfdate.' '.$strftime);
            /* ip = u->ipaddress;
            aux = arr(usertext, roletext, start, end, ip); */

            //convert user lesson duration sec to HH:mm:s
            $lessonduration = null;
            if(isset($u->duration)) {
                $lessonUserDuration = gmdate("H:i:s", $u->duration);
            }

            //check if codice fiscale exist then push codicefiscale
            if(!empty($codicefiscalecode)){

                //check user lesson session id then push to array
                if(isset($lessonsessionid)){

                    //check user lesson total duration
                    if(isset($total) && $total == 'total'){
                        $aux = array($usertext, $roletext,$u->codicefiscale, $COURSE->fullname, $cm->name, $lessonUserDuration);
                    }else{
                        $aux = array($usertext, $roletext,$u->codicefiscale, $COURSE->fullname, $cm->name, $start, $end , $lessonUserDuration);
                    }

                }else{

                    $aux = array($usertext, $roletext,$u->codicefiscale, $COURSE->fullname, $cm->name, $start, $end);

                }

            }else{

                // //check user lesson session id then push to array
                if(isset($lessonsessionid)){

                    //check user lesson total duration
                    if(isset($total) && $total == 'total'){
                        $aux = array($usertext, $roletext,$COURSE->fullname, $cm->name, $lessonUserDuration);
                    } else{
                        $aux = array($usertext, $roletext,$COURSE->fullname, $cm->name, $start, $end , $lessonUserDuration);
                    } 

                }else{

                    $aux = array($usertext, $roletext,$COURSE->fullname, $cm->name, $start, $end);

                }

            }
            
            array_push($tableuserssessions->data, $aux);

        }

    }

    return $tableuserssessions;
}

/**
 * This function build an html table using a Moodle table object as input
 *
 * @param object $table Html table object
 * @return string $html Html table
 */
function tlive_buildhtmltable($table) {

    $html="<p></p>
    <table border=\"1\">
    <tr style=\"font-size: large;\" bgcolor=\"#cc1111\">";

    foreach ($table->head as $h) {
        $html.="<td>$h</td>";
    }

    $html.='</tr></thead>
    <tbody>';
    $count=1;

    foreach ($table->data as $data) {

        if ($count % 2==0) {
            $color = "#bce7ff";
        } else {
            $color = "#ffffff";
        }

        $html.="<tr bgcolor=\"$color\">";

        foreach ($data as $d) {
            $html.="<td>$d</td>";
        }

        $html.="</tr>";

        $count++;
    }
    $html.="</table>";

    return $html;
}


/**
 * This function is called by task scheduling and checks for each room, if is necessary to update its statistics
 * @param int $daylimit threshold in days, beyond which is not necessary to update room statistics
 */
function tlive_task_updatestats($daylimit) {

    global $DB;
    /* Don't update rooms that have finished from more than two days */

    $today = time();
    $diff = $today - $daylimit*3600*24;

    $query = 'SELECT * FROM {tlive} WHERE enddate>=? AND startdate<=?';

    if ($rooms = $DB->get_records_sql($query, array($diff, $today) ) ) {
        /* Check for each room */
        foreach ($rooms as $room) {

            if (! $course = $DB->get_record('course',  array('id' => $room->course))) {
                print_error('Course is misconfigured');
            }

            if (! $cm = get_coursemodule_from_instance('tlive',  $room->id,  $course->id)) {
                print_error('Course Module ID was incorrect');
            }

            mtrace('Updating stats...');       
            /* Update rooms statistics and activities completion */
            tlive_updatestats($room, $course, $cm);
            mtrace('Finished updating stats');

        }

    }
}


/**
 * This function update room logs and activity completion for each user
 *
 * @param object $tlive Tlive instance object
 * @param object $course Course instance
 * @param object $cm object Course module instance
 */
function tlive_updatestats($tlive, $course, $cm) {

    global $CFG, $DB;
    /* Required to fix bug -> Fatal error: Class 'completion_info' not found  */
    require_once("$CFG->libdir/completionlib.php");

    $event = \mod_tlive\event\stats_update::create(array(
	    'objectid' => $tlive->id,
	    'context' => context_module::instance($cm->id)
	));
	$event->trigger();

    $complete = 0;
    $xmlanswer = tlive_askforlogs($complete,  $tlive);
    tlive_savelogs($xmlanswer);
    /* Updating activity completion for all participants */
    $completion = new completion_info($course);

    if ($completion->is_enabled($cm) && ($tlive->completiontlive || $tlive->completiontlivepoll) ) {

        /*  query to extract users from lessions
            $query = 'SELECT DISTINCT us.iduser, rs.roomid, ra.roleid FROM mdl_tlive_lesson_user_sessions as uls
            INNER JOIN mdl_tlive_user_sessions as us ON uls.usersessionid=us.usersessionid
            INNER JOIN mdl_tlive_rooms_sessions as rs ON us.roomsessionid=rs.roomsessionid
            INNER JOIN mdl_role_assignments as ra ON us.iduser=ra.userid
            WHERE rs.roomid=? AND ra.roleid=5';
        */
        /*  query to extract users from room sessions -->if and only if user is a student (roleid=5),
            check activity as completed
            $query = 'SELECT DISTINCT us.iduser FROM mdl_tlive_user_sessions as us
            INNER JOIN mdl_tlive_rooms_sessions as r ON us.roomsessionid = r.roomsessionid
            INNER JOIN mdl_role_assignments as ra ON us.iduser=ra.userid
            WHERE r.roomid=? AND ra.contextid=? AND ra.roleid=5';
        */
        /* query to extract users relative to Teleskill Live role
            -> if and only if user is not a conference managae */
            $query = 'SELECT DISTINCT us.iduser FROM {tlive_user_sessions} as us
            INNER JOIN {tlive_rooms_sessions} as r ON us.roomsessionid = r.roomsessionid
            WHERE r.roomid=?';

        if ($users = $DB->get_records_sql($query, array($tlive->roomid, $tlive->confmanid))) {
            foreach ($users as $user) {

                /* $user_role = get_user_roles($context, $user->iduser);
                updating activity completion for each user */
                $completion->update_state($cm, COMPLETION_COMPLETE, $user->iduser);

            }
        }
    }
}
