<?php

if(file_exists(__DIR__ . '/../vendor/autoload.php')) {
	require_once(__DIR__ . '/../vendor/autoload.php');
}

use avadim\FastExcelWriter\Excel;

class TvclHelper {
	
	private $ccc;
	private $tvclurl;

	public function __construct() {
		$settings = get_option('tlive-settings');
		$this->tvclurl = $settings['tvcl_url'];
		$this->ccc = $settings['ccc'];
	}

	public function createXML($type) {
		//-->Begin XML creation<--
		$XML= new SimpleXMLElement("<ews></ews>");
		$ccc = $this->getCCC();
		$XML->addAttribute('type',$type);
		$XML->addChild('clientcode',$ccc);
		return $XML;
		//--->End XML creation<----
	}

	public function addRoomFields($data, $XML) {
		$timezone = new DateTimeZone(wp_timezone_string());
		$title = sanitize_text_field($data['title']);
		$users = (int)$data['usersNumber'];
		
		/* build unixtime timestamp (UTC), considering cms timezone */
		$settings = get_option('tlive-settings');
		
		/* if room scheduling is enabled */
		if (!$settings['disable_date']) {
			$day = $data['day'];
			$start = $data['start'];
			$end = $data['end'];
			$datestart = new DateTime($day . ' ' . $start, $timezone);
			$dateend = new DateTime($day . ' ' . $end, $timezone);
	
			/* convert to UTC */
			$datestart->setTimeZone(new DateTimeZone('UTC'));
			$dateend->setTimeZone(new DateTimeZone('UTC'));

			$XML->addChild('startdate', $datestart->format('Y-m-d H:i:s'));
			$XML->addChild('enddate', $dateend->format('Y-m-d H:i:s'));
		}
		
		$XML->addChild('title', $title);
		$XML->addChild('users', $users);

		$upload_dir = wp_upload_dir();

		if ($data['logo'] != null) {
			$logo = $XML->addChild('logo', $upload_dir['url'] . '/' . $data['logo']);
		}

		if ($data['banner'] != null) {
			$logo = $XML->addChild('banner', $upload_dir['url'] . '/' . $data['banner']);
		}
	}

	public function createRoom($data) {
		$XML = $this->createXML(1);
		$this->addRoomFields($data, $XML);
		$XMLResponse = $this->getXMLResponse($XML);
		return $XMLResponse;
	}

	public function editRoom($data, $roomid) {
		$XML = $this->createXML(6);
		$this->addRoomFields($data, $XML);
		$XML->addChild('roomid', $roomid);
		$XMLResponse = $this->getXMLResponse($XML);
		return $XMLResponse;
	}

	public function deleteRoom($roomid) {
		$XML = $this->createXML(2);
		$XML->addChild('roomid', $roomid);
		$XMLResponse = $this->getXMLResponse($XML);
		return $XMLResponse;
	}

	public function enableUser($data, $roomid) {
		$name = (!empty($data->first_name) || !empty($data->last_name)) ? $data->first_name . ' ' . $data->last_name : $data->name;
		
		$XML = $this->createXML(3);
		$XML->addChild('roomid', $roomid);
		$XML->addChild('lmsuserid', $data->userid);
		$XML->addChild('role', $data->role);
		$XML->addChild('name', $name);
		$XML->addChild('firstname', $data->first_name);
		$XML->addChild('lastname', $data->last_name);
		$XML->addChild('email', $data->email);
		$XML->addChild('lang', $data->lang);
		$XMLResponse = $this->getXMLResponse($XML);
		return $XMLResponse;
	}

	public function getRoomLogs($roomid) {
		$XML = $this->createXML(16);
		$XML->addChild('roomid', $roomid);
		$XML->addChild('full', 1);
		$XMLResponse = $this->getXMLResponse($XML);
		return $XMLResponse;
	}

	public function getXMLResponse($XML) {
		//Send XML request and receive server response
		$ch = curl_init($this->getTvclURL());
		curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type: text/xml'));
		curl_setopt($ch, CURLOPT_HEADER, 0);
		curl_setopt($ch, CURLOPT_POST, 1);
		curl_setopt($ch, CURLOPT_POSTFIELDS, $XML->asXML());
		curl_setopt($ch, CURLOPT_FOLLOWLOCATION, 0);
		curl_setopt($ch, CURLOPT_REFERER, 'http://www.teleskill.net');
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
		$ch_response = curl_exec($ch);		
		$ch_info = curl_getinfo($ch);		
		
		if ($ch_info['http_code'] != 200) {
			$ch_response = '';
		}

		//print_r($XML->asXML());
		curl_close($ch);
		$XMLResponse = simplexml_load_string($ch_response);

		//echo $XML->asXML();
		return $XMLResponse;
	}

	public function getCCC() {
		return $this->ccc;
	}

	public function setCCC($ccc) {
		$this->ccc = $ccc;
	}

	public function getTvclURL() {
		return $this->tvclurl;
	}

	function saveLogs($xmlstructure) {

		global $wpdb;

	    //$xmlstructure = simplexml_load_string($xmlanswer);
	    /* Statistics are passed by Tlive in Europe/Rome local time*/
	    date_default_timezone_set('Europe/Rome');

	    if ($xmlstructure->Room) {
	        $room = $xmlstructure->Room;
	        $roomid = (int) $xmlstructure->Room->attributes()->ID;
	        $query = "SELECT * FROM $wpdb->tlive_rooms_logs WHERE roomid = %d";

	        if (!$result = $wpdb->get_row($wpdb->prepare($query, $roomid))) {
	            $query = "INSERT INTO $wpdb->tlive_rooms_logs VALUES (%d, %d)";
	            if (!$wpdb->query($wpdb->prepare($query, $roomid, 0)) ) {
            		return false;
            	}
	        }

	        foreach ($room->RoomSession as $roomsession) {

	            $roomsessionid = (int)$roomsession->attributes()->ID;
	            $startdate = strtotime($roomsession->attributes()->StartDate);
	            $startdate = gmdate('Y-m-d H:i:s', $startdate);
	            $enddate = strtotime($roomsession->attributes()->EndDate);
	            $enddate = gmdate('Y-m-d H:i:s', $enddate);

	            $query = "SELECT * FROM $wpdb->tlive_rooms_sessions WHERE roomsessionid = %d";

	            if (!$result = $wpdb->get_row($wpdb->prepare($query, $roomsessionid))) {

	                $query = "INSERT INTO $wpdb->tlive_rooms_sessions VALUES (%d, %d, %s, %s, %d)";

	                /* Notice: you can't use insert_record because primary key (roomid) is already defined,
	                 * use execute SQL */
	                if(!$wpdb->query($wpdb->prepare($query, $roomsessionid, $roomid, $startdate, $enddate, 0) ) ) {
            			return false;
            		}
	            }

	            /* if polls sessions exist, save them */
	            if ($roomsession->Polls) {
	               	if( !$this->getpolls($roomsession->Polls, $roomsessionid)) {
	               		return false;
	            	}
	            }

	            /* if users sessions exist, save them */
	            if ($roomsession->UserSessions) {
	                if(!$this->getusersessions($roomsession->UserSessions, $roomsessionid)) {
	                	return false;
	                }
	            }

	            /* if lessions sessions exist, save them */
	            if ($roomsession->LessonSessions) {
	               if (!$this->getlessonsessions($roomsession->LessonSessions, $roomsessionid)) {
	               		return false;
	               }
	            }
           	}
		}

		date_default_timezone_set(wp_timezone_string());
		return true;
	}

	/**
	 * This function saves polls data to db, from polls simplexml object
	 *
	 * @param object $polls Polls data simplexml object
	 * @param int $roomsessionid Room session id
	 */
	function getpolls($polls, $roomsessionid) {

	    global $wpdb;

	    foreach ($polls->Poll as $poll) {

	        $pollid = (int)$poll->attributes()->ID;
	        /* replace "?" and prevent any quotes */
	        $question = addslashes(str_replace("?", '', $poll->attributes()->Question));

	        $query = "SELECT * FROM $wpdb->tlive_polls WHERE pollid = %d";

	        if (!$result = $wpdb->get_row($wpdb->prepare($query, $pollid))) {
	            $query="INSERT INTO $wpdb->tlive_polls VALUES (%d, %s, %d)";
	            if(!$wpdb->query($wpdb->prepare($query, $pollid, $question, $roomsessionid) ) ) {
	            	return false;
	            }
	        }

	        if ($poll->Answers) {
	            if (!$this->getanswers($poll->Answers, $pollid)) {
	            	return false;
	            }
	        }

	        if ($poll->PollSessions) {
	        	if(!$this->getpollsessions($poll->PollSessions, $pollid)) {
	        		return false;
	        	}
	        }
	    }
	    return true;
	}

	/**
	 * This function saves poll answers data to db, from answers simplexml object
	 *
	 * @param object $answers Answers simplexml object
	 * @param int $pollid Poll id
	 */
	function getanswers($answers, $pollid) {

	    global $wpdb;

	    foreach ($answers->Answer as $answer) {
	        $answerid = (int)$answer->attributes()->ID;
	        $sequence = (int)$answer->attributes()->Sequence;
	        $answerstr = addslashes(str_replace("?", '', $answer->attributes()->Answer));
	        $correct = (int)$answer->attributes()->Correct;
	        $query = "SELECT * FROM $wpdb->tlive_polls_answers WHERE answerid = %d";

	        if (!$result = $wpdb->get_row($wpdb->prepare($query,$answerid) ) ) {
	            $query = "INSERT INTO $wpdb->tlive_polls_answers VALUES (%d, %d, %s, %d, %d)";
	            if(!$wpdb->query($wpdb->prepare($query, $answerid, $pollid, $answerstr, $sequence, $correct) )) {
	            	return false;
	            }
	        }
	    }
	    return true;
	}

	/**
	 * This function saves poll sessions data to db, from answers simplexml object
	 *
	 * @param object $pollsessions Answers simplexml object
	 * @param int $pollid Poll id
	 */
	function getpollsessions($pollsessions, $pollid) {
	   global $wpdb;
	   foreach ($pollsessions->PollSession as $pollsession) {
	        $pollsessionid = (int)$pollsession->attributes()->ID;
	        $opendate = (int)strtotime($pollsession->attributes()->OpenDate);
	        $opendate = gmdate('Y-m-d H:i:s', $opendate);
	        $closedate = $pollsession->attributes()->CloseDate;

	        /* if close date is null, save room session's end as close date */
	        if (empty($closedate)) {
	        	$query = "SELECT rs.enddate FROM $wpdb->tlive_polls AS p
							INNER JOIN  $wpdb->tlive_rooms_sessions AS rs ON p.roomsessionid=rs.roomsessionid
							WHERE p.pollid=%d";
				$closedate = $wpdb->get_var($wpdb->prepare($query, $pollid) );
	        } else {
	        	$closedate = (int)strtotime($closedate);
	        	$closedate = gmdate('Y-m-d H:i:s', $closedate);
	        }

	        $query = "SELECT * FROM $wpdb->tlive_polls_sessions WHERE pollsessionid=%d";
	        if (!$result = $wpdb->get_row($wpdb->prepare($query,$pollsessionid) ) ) {
	            $query="INSERT INTO $wpdb->tlive_polls_sessions VALUES (%d, %d, %s, %s)";

	            if (!$wpdb->query($wpdb->prepare($query,$pollsessionid, $pollid, $opendate, $closedate ) ) ) {
	            	//echo $db->getErrorMsg();
	            	return false;
	            }
	        }

	        $useranswers = $pollsession->UserAnswer;
	        if (!$this->getuseranswers($useranswers, $pollsessionid) ) {
	        	return false;
	        }
	    }
	    return true;
	}

	/**
	 * This function saves user answers data to db, from user answers simplexml object
	 *
	 * @param object $useranswers Answers simplexml object
	 * @param int $pollsessionid Poll session id
	 */
	function getuseranswers($useranswers, $pollsessionid) {

	    global $wpdb;

	    foreach ($useranswers as $useranswer) {

	        $useranswerid = (int)$useranswer->attributes()->ID;
	        $usersessionid = (int)$useranswer->attributes()->UserSessionID;
	        $answerid = (int)$useranswer->attributes()->AnswerID;
	        $creationdate = strtotime($useranswer->attributes()->CreationDate);
	        $creationdate = gmdate('Y-m-d H:i:s', $creationdate);

			$query = "SELECT * FROM $wpdb->tlive_users_poll_answers WHERE userspollanswerid=%d";

	        if (!$result = $wpdb->get_row($wpdb->prepare($query, $useranswerid ) ) ) {
	            $query="INSERT INTO $wpdb->tlive_users_poll_answers
	                    VALUES (%d, %d, %d, %d, %s)";

	            if(!$wpdb->query($wpdb->prepare($query, $useranswerid, $pollsessionid, $usersessionid, $answerid, $creationdate ) ) ) {
	            	return false;
	            }
	        }
	    }
	    return true;
	}

	/**
	 * This function saves user sessions data to db, from user sessions simplexml object
	 *
	 * @param object $usersessions Answers simplexml object
	 * @param int $roomsessionid Room session id
	 */
	function getusersessions($usersessions, $roomsessionid) {

	    global $wpdb;

	    foreach ($usersessions->UserSession as $usersession) {

	        $usersessionid = (int)$usersession->attributes()->ID;
	        $userid = (int)$usersession->attributes()->UserID;
	        $roleid = (int)$usersession->attributes()->RoleID;
	        $startdate = strtotime($usersession->attributes()->StartDate);
	        $startdate = gmdate('Y-m-d H:i:s', $startdate);

	        $enddate = strtotime($usersession->attributes()->EndDate);
	        $enddate = gmdate('Y-m-d H:i:s', $enddate);

			$query = "SELECT * FROM $wpdb->tlive_user_sessions WHERE usersessionid = %d";

	        if (!$result = $wpdb->get_row($wpdb->prepare($query, $usersessionid) ) ) {
	            $query = "INSERT INTO $wpdb->tlive_user_sessions VALUES (%d, %d, %d, %d, %s, %s, %s)";

	            if (!$wpdb->query($wpdb->prepare($query, $usersessionid, $userid, $roomsessionid, $roleid, $startdate, $enddate, '')) ) {
	            	return false;
	            }
	        }
	    }
	    return true;
	}

	/**
	 * This function saves lesson sessions data to db, from lesson sessions simplexml object
	 *
	 * @param object $lessonsessions Lesson sessions simplexml object
	 * @param int $roomsessionid Room session id
	 */
	function getlessonsessions($lessonsessions, $roomsessionid) {

	    global $wpdb;

	    foreach ($lessonsessions->LessonSession as $lessonsession) {

	        $lessonsessionid = (int) $lessonsession->attributes()->ID;
	        $lessonstartdate = strtotime($lessonsession->attributes()->StartDate);
	        $lessonstartdate = gmdate('Y-m-d H:i:s', $lessonstartdate);
	        $lessonenddate = strtotime($lessonsession->attributes()->EndDate);
	        $lessonenddate = gmdate('Y-m-d H:i:s', $lessonenddate);
	        $lessonduration = $lessonsession->attributes()->Duration;

	        $query = "SELECT * FROM $wpdb->tlive_lesson_sessions WHERE lessonsessionid = %d";

	        if (!$result = $wpdb->get_row($wpdb->prepare($query, $lessonsessionid) ) ) {
	            $query = "INSERT INTO $wpdb->tlive_lesson_sessions VALUES (%d, %d, %s, %s, %d)";
	            if (!$wpdb->query($wpdb->prepare($query, $lessonsessionid, $roomsessionid, $lessonstartdate, $lessonenddate, $lessonduration))) {
	            	return false;
	            }
	        }

			foreach ($lessonsession->PresenceCheckList->PresenceCheck as $presenceCheck) {
				
				$timeCheck = strtotime($presenceCheck->attributes()->TimeCheck);
				$timeCheck = gmdate('Y-m-d H:i:s', $timeCheck);
				
				$query = "INSERT INTO $wpdb->tlive_presence_check VALUES (%d, %d, %s)";
				if (!$wpdb->query($wpdb->prepare($query, (int)$presenceCheck->attributes()->ID, $lessonsessionid, $timeCheck))) {
	            	return false;
	            }

				if($presenceCheck->User) {
					foreach ($presenceCheck->User as $user) {

						$userTimeCheck = strtotime($user->attributes()->TimeCheck);
						$userTimeCheck = gmdate('Y-m-d H:i:s', $userTimeCheck);

						$query = "INSERT INTO $wpdb->tlive_user_presences VALUES (%d, %d, %d, %s)";
						if (!$wpdb->query($wpdb->prepare($query, (int)$user->attributes()->ID, (int)$presenceCheck->attributes()->ID, (int)$user->attributes()->UserSessionID, $userTimeCheck))) {
							return false;
						}
					}
				}
			}

	        if (!$this->setuserlessonsession($lessonsession, $roomsessionid)) {
	        	return false;
	        }
	    }
	    return true;
	}

	/**
	 * This function compute the effective user lesson session, crossing user sessions data with lesson sessions data
	 * and save it to db
	 *
	 * @param object $lessonsession Lesson session simplexml object
	 * @param int $roomsessionid Room session id
	 */
	function setuserlessonsession($lessonsession, $roomsessionid) {

	    global $wpdb;

	    $lessonstartdate = strtotime($lessonsession->attributes()->StartDate);
	    $lessonenddate   = strtotime($lessonsession->attributes()->EndDate);
	    $lessonsessionid = (int)$lessonsession->attributes()->ID;

	    /* extract all users sessions */
	    $query = "SELECT * FROM $wpdb->tlive_user_sessions WHERE roomsessionid = %d";

	    if ($usersessions = $wpdb->get_results($wpdb->prepare($query, $roomsessionid) ) ) {


	        /* for each user session check if it intersects lesson session, if true, save it to db */
	        foreach ($usersessions as $usersession) {

				date_default_timezone_set('UTC');
				$usersession_startdate = strtotime($usersession->startdate);
				$usersession_enddate = strtotime($usersession->enddate);

				date_default_timezone_set('Europe/Rome');
	            $cond = ($lessonenddate >= $usersession_startdate && $lessonstartdate <= $usersession_enddate );

	            if ($cond) {

	                /* save user lesson session  */
	                $userlessonsessionstart = max($lessonstartdate, $usersession_startdate);
	                $userlessonsessionend = min($lessonenddate, $usersession_enddate);
	                $usersessionid = $usersession->usersessionid;
	                $duration = $userlessonsessionend - $userlessonsessionstart;
	                $userlessonsessionstart = gmdate('Y-m-d H:i:s', $userlessonsessionstart);
	                $userlessonsessionend = gmdate('Y-m-d H:i:s', $userlessonsessionend);

					$query = "SELECT * FROM $wpdb->tlive_lesson_user_sessions WHERE lessonsessionid = %d AND usersessionid = %d";

	                if (!$check = $wpdb->get_row($wpdb->prepare($query, $lessonsessionid, $usersessionid) ) ) {
	                	$query = "INSERT INTO $wpdb->tlive_lesson_user_sessions VALUES (%d, %d ,%d, %s, %s,%d)";
	                    if(!$wpdb->query($wpdb->prepare($query, 0, $lessonsessionid, $usersessionid, $userlessonsessionstart, $userlessonsessionend, $duration))) {
	            			return false;
	            		}
	                }
	            }
	        }
	    }
	    return true;
	}

	public function buildTableTotal($id) {

		global $wpdb;

		$heading_user = __('Participant name', 'teleskill-tlive');
	    $heading_start = __('Start', 'teleskill-tlive');
	    $heading_end =  __('End', 'teleskill-tlive');
		$duration = __('Duration', 'teleskill-tlive');
		$totalCheck = __('Total check', 'teleskill-tlive');
		$passedCheck = __('Passed check', 'teleskill-tlive');;

		$exportTable = new stdClass();
		$exportTable->class = 'user_sessions';
		$exportTable->head  = [$heading_user, $heading_start, $heading_end, $duration, $totalCheck, $passedCheck];
		$exportTable->rowclasses = [];
		$exportTable->data = [];

		$query = "SELECT * FROM $wpdb->tlive_lesson_sessions AS rs WHERE rs.lessonsessionid=%d";
		$roomSession = $wpdb->get_row($wpdb->prepare($query, $id));

		$query = "SELECT us.iduser, MIN(uls.startdate) as start, MAX(uls.enddate) as end, SUM(uls.duration) as duration
			FROM $wpdb->tlive_lesson_user_sessions AS uls
			INNER JOIN $wpdb->tlive_user_sessions AS us ON uls.usersessionid = us.usersessionid			
			WHERE uls.lessonsessionid = %d group by us.iduser";
		$exportTable->title = __('Lesson Access List Total', 'teleskill-tlive');
		$lessonData = $wpdb->get_results($wpdb->prepare($query, $id), ARRAY_A);

		// Recupero il totale delle verifiche di presenza
		$query = "SELECT COUNT(*) as num_rows FROM $wpdb->tlive_presence_check WHERE lesson_session_id = %d";
		$checkCount = $wpdb->get_results($wpdb->prepare($query, $id));

		// Recupero il dettaglio delle verifiche per singolo utente
		$query = "SELECT us.iduser, Count(*) AS PresenceCheckCount
					FROM wp_tlive_presence_check pc
					INNER JOIN wp_tlive_user_presences up ON pc.presence_id = up.presence_id
					INNER JOIN wp_tlive_user_sessions us ON up.user_session_id = us.usersessionid
					WHERE pc.lesson_session_id = %d group by us.iduser";
		$checkCountDetail = $wpdb->get_results($wpdb->prepare($query, $id), ARRAY_A);
		
		for($i = 0; $i < count($lessonData); $i++) {
			$userinfo = get_userdata($lessonData[$i]['iduser']);
			$key = array_search($lessonData[$i]['iduser'], array_column($checkCountDetail, 'iduser'));
			
			/* format date event, time start and close */
			$lessonData[$i]['participant'] = (!empty($userinfo->first_name) || !empty($userinfo->last_name)) ? $userinfo->first_name . ' ' . $userinfo->last_name : $userinfo->user_nicename;	 
			$lessonData[$i]['start'] = tlive_print_datetime($lessonData[$i]['start'], 'l, j F Y H:i');
			$lessonData[$i]['end'] = tlive_print_datetime($lessonData[$i]['end'], 'l, j F Y H:i');
			$lessonData[$i]['duration'] = gmdate("H:i:s", $lessonData[$i]['duration']);
			$lessonData[$i]['total_check'] = (int)$checkCount[0]->num_rows;
			$lessonData[$i]['passed_check'] = (is_numeric($key)) ? $checkCountDetail[$key]["PresenceCheckCount"] : 'Conference Manager';

			$aux = array($lessonData[$i]['participant'], $lessonData[$i]['start'], $lessonData[$i]['end'], $lessonData[$i]['duration'], $lessonData[$i]['total_check'], $lessonData[$i]['passed_check']);
			array_push($exportTable->data, $aux);
		}

		return $exportTable;

	}
	public function buildtableaccess($type, $id) {

 		global $wpdb;

	    $heading_user = __('Participant name', 'teleskill-tlive');
	    $heading_start = __('Start', 'teleskill-tlive');
	    $heading_end =  __('End', 'teleskill-tlive');

		$table_users_sessions = new stdClass();
		$table_users_sessions->class = 'user_sessions';
		$table_users_sessions->head  = [$heading_user, $heading_start, $heading_end];
		$table_users_sessions->rowclasses = [];
		$table_users_sessions->data = [];

		// Two types of sessions: Lesson or Room sessions
		switch($type) {
			case 'lessonaccess':
				$query = "SELECT * FROM $wpdb->tlive_lesson_sessions AS rs WHERE rs.lessonsessionid=%d";
				$roomsession = $wpdb->get_row($wpdb->prepare($query, $id));

				$query = "SELECT uls.sessionid, us.iduser, uls.startdate, uls.enddate, us.roleid, us.ipaddress, ls.startdate AS sessionstart,ls.enddate AS sessionend
						FROM $wpdb->tlive_lesson_user_sessions AS uls
						INNER JOIN $wpdb->tlive_user_sessions AS us ON uls.usersessionid=us.usersessionid
						INNER JOIN $wpdb->tlive_lesson_sessions AS ls ON uls.lessonsessionid=ls.lessonsessionid
						WHERE uls.lessonsessionid=%d";
				$table_users_sessions->title = __('Lesson Access List', 'teleskill-tlive');
				break;
			case 'roomaccess':
				$query = "SELECT * FROM $wpdb->tlive_rooms_sessions AS rs WHERE rs.roomsessionid=%d";
				$roomsession = $wpdb->get_row($wpdb->prepare($query, $id));

				$query = "SELECT us.iduser, us.startdate,us.enddate,rs.startdate AS sessionstart,rs.enddate AS sessionend
							FROM $wpdb->tlive_user_sessions AS us
							INNER JOIN $wpdb->tlive_rooms_sessions AS rs ON us.roomsessionid=rs.roomsessionid
							WHERE us.roomsessionid=%d";
				$table_users_sessions->title = __('Rooms Access List', 'teleskill-tlive');
				break;
		}

	  	if ($user_sessions = $wpdb->get_results($wpdb->prepare($query, $id) ) ) {
			$sessiondate = tlive_print_datetime($roomsession->startdate, 'l, j F Y');
			$start = tlive_print_datetime($roomsession->startdate, 'H:i');
		    $end = tlive_print_datetime($roomsession->enddate, 'H:i');
			$sessiontext = sprintf(__('Session of %s from %s to %s', 'teleskill-tlive'), $sessiondate, $start, $end);

	    	$table_users_sessions->title .= ' - ' . $sessiontext;

	        foreach ($user_sessions as $u) {
	        	$userinfo = get_userdata($u->iduser);
				/* format date event, time start and close */
				$user_text = (!empty($userinfo->first_name) || !empty($userinfo->last_name)) ? $userinfo->first_name . ' ' . $userinfo->last_name : $userinfo->user_nicename;	 
				$start = tlive_print_datetime($u->startdate , 'l, j F Y H:i');
				$end = tlive_print_datetime($u->enddate , 'l, j F Y H:i');
	        	$aux = array($user_text, $start, $end);
	        	array_push($table_users_sessions->data, $aux);
	        }
		}
		return $table_users_sessions;
 	}

 	public function buildtablepolls($id) {

 		global $wpdb;

	    //table heading
	    $heading_user = __('Participant\'s Name', 'teleskill-tlive');
	    $heading_answer =  __('Answer', 'teleskill-tlive');
	    $heading_correct = __('Result', 'teleskill-tlive');

	    $table = new stdClass();
		$table->class = 'user_answers';
		$table->head  = array($heading_user, $heading_answer, $heading_correct);
		$table->rowclasses = [];
		$table->data = [];

		$query = "SELECT upa.userspollanswerid, us.iduser, ps.correct, us.roleid,
	     	 ps.answer, us.startdate, us.enddate
	      	FROM ($wpdb->tlive_users_poll_answers AS upa
	        INNER JOIN $wpdb->tlive_user_sessions AS us ON upa.usersessionid=us.usersessionid
	        INNER JOIN $wpdb->tlive_polls_answers AS ps ON upa.answerid=ps.answerid
	        )
	        INNER JOIN $wpdb->users as u ON us.iduser = u.id
	        WHERE pollsessionid=%d";

 		if ($result = $wpdb->get_results($wpdb->prepare($query, $id))) {

 			$query = "SELECT p.question, ps.opendate, ps.closedate FROM $wpdb->tlive_polls_sessions AS ps
					INNER JOIN $wpdb->tlive_polls AS p ON ps.pollid=p.pollid
					WHERE ps.pollsessionid=%d";

			$pollsession = $wpdb->get_row($wpdb->prepare($query, $id));
 			$sessiondate = tlive_print_datetime($pollsession->opendate, 'l, j F Y');
			$start = tlive_print_datetime($pollsession->opendate, 'H:i');
	   		$end = tlive_print_datetime($pollsession->closedate, 'H:i');
			$sessiontext = __('Question', 'teleskill-tlive') . ': ' . stripslashes($pollsession->question) . ', ' . sprintf(__('Session of %s from %s to %s', 'teleskill-tlive'), $sessiondate, $start, $end);
 			$table->title = $sessiontext;

 			foreach ($result as $r) {
 				$userinfo = get_userdata($r->iduser);
				/* format date event, time start and close */
				$user_text = (!empty($userinfo->first_name) || !empty($userinfo->last_name)) ? $userinfo->first_name . ' ' . $userinfo->last_name : $userinfo->user_nicename;	 

 				if($r->correct == 1){
 					$correct = __('Correct', 'teleskill-tlive');
				}else {
					$correct = __('Wrong', 'teleskill-tlive');
				}

 				$row = array($user_text, $r->answer, $correct);
 				array_push($table->data, $row);
 			}
 		}
	 	return $table;
 	}

 	public function buildhtmltable($table) {

   		$html = "<p></p><table border=\"1\"><tr style=\"font-size: large;\" bgcolor=\"#cc1111\">";

   		foreach ($table->head as $h){
   			$html .= "<td>$h</td>";
   		}

   		$html .= '</tr></thead><tbody>';
   		$count = 1;

   		foreach ($table->data as $data){

   			if($count % 2 == 0){
   				$color = "#bce7ff";
   			} else {
   				$color = "#ffffff";
   			}

   			$html .= "<tr bgcolor=\"$color\">";

   			foreach($data as $d){
	   			$html .= "<td>$d</td>";
	   		}

	   		$html .= "</tr>";

   			$count++;
   		}

   		$html .= "</table>";

   		return $html;
   	}

	public function exportXls($table, $fileName) {
		
		$outFileName = __DIR__ . '/output/' . $fileName . '.xlsx';
		
		$header = [];
		$dataRow = [];

		foreach ($table->head as $h) {
			$header[] = $h;
		}

		foreach($table->data as $data) {
			var_dump($data);
			echo "<br>";
			$dataRow[] = $data;
		}
		echo "<hr>";
		// Create Excel workbook
		$excel = Excel::create();
		
		// Get the first sheet;
		$sheet = $excel->getSheet();
		
		$rowOptions = ['font-style' => 'bold'];
		// Write header:
		//    $header - values for cells of the first row and options for columns
		//    $rowOptions - options of header row
		$sheet->writeHeader($header, $rowOptions);
		
		// The fastest way to write data is row by row
		foreach($dataRow as $row) {
			var_dump($row);
			echo "<br>";
			$sheet->writeRow($row);
		}
		
		// Save to XLSX-file
		//$excel->save($outFileName);
		ob_get_clean();
		$excel->download($outFileName);
		ob_end_flush();
        die();

	}

	public function exportPdf($table, $type = '') {

		require_once('fpdf/fpdf.php');

		switch ($type) {
			case POLLS_LIST_STATS:
			case SESSIONS_LIST_STATS:
			case LESSONS_LIST_STATS:
				$orientation = 'P';
				$columnNum = 3;
				$tableWidth = 180;
				$cellWidth = 60;
				break;
			case LESSONS_LIST_TOTAL_STATS:
				$orientation = 'L';
				$columnNum = 6;
				$tableWidth = 270;
				$cellWidth = 45;
				break;
			default:
				$orientation = 'P';
				$columnNum = 3;
				$tableWidth = 180;
				$cellWidth = 60;
				break;
		}

		$pdf = new FPDF($orientation);
		$pdf->AddPage();
		
		// Colors, line width and bold font
		$pdf->SetFillColor(255,0,0);
		$pdf->SetTextColor(255);
		$pdf->SetDrawColor(128,0,0);
		$pdf->SetLineWidth(.3);
		$pdf->SetFont('Arial', '', 14);
		// Header
		$w = array(40, 35, 40, 45);
		for($i = 0; $i < $columnNum; $i++) {
			$pdf->Cell($cellWidth, 7, $table->head[$i], 1, 0, 'C', true);
		}
		$pdf->Ln();
		
		$pdf->SetFillColor(224, 235, 255);
		$pdf->SetTextColor(0);
		$pdf->SetFont('');
		// Data
		$fill = false;
		$pdf->SetFont('Arial', '', 8);
		foreach($table->data as $row) {
			$pdf->Cell($cellWidth, 6, mb_convert_encoding($row[0], 'windows-1252', 'UTF-8'), 'LR', 0, 'L', $fill);
			$pdf->Cell($cellWidth, 6, mb_convert_encoding($row[1], 'windows-1252', 'UTF-8'), 'LR', 0, 'L', $fill);
			$pdf->Cell($cellWidth, 6, mb_convert_encoding($row[2], 'windows-1252', 'UTF-8'), 'LR', 0, 'L', $fill);
			if($columnNum === 6) {
				$pdf->Cell($cellWidth, 6, mb_convert_encoding($row[3], 'windows-1252', 'UTF-8'), 'LR', 0, 'L', $fill);
				$pdf->Cell($cellWidth, 6, mb_convert_encoding($row[4], 'windows-1252', 'UTF-8'), 'LR', 0, 'L', $fill);
				$pdf->Cell($cellWidth, 6, mb_convert_encoding($row[5], 'windows-1252', 'UTF-8'), 'LR', 0, 'L', $fill);
			}
			
			$pdf->Ln();
			$fill = !$fill;
		}
		// Closing line
		$pdf->Cell($tableWidth, 0, '', 'T');


		$pdf->Output();
	}

 	public function exportpdf_OLD($table){
   		//caricamento libreria tcpdf
   		require_once('libraries/tcpdf/tcpdf.php');
   		require_once('libraries/tcpdf/lang/ita.php');

   		// create new PDF document
		$pdf = new TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);
		
		// set header and footer fonts
		$pdf->setHeaderFont(Array(PDF_FONT_NAME_MAIN, '', PDF_FONT_SIZE_MAIN));
		$pdf->setFooterFont(Array(PDF_FONT_NAME_DATA, '', PDF_FONT_SIZE_DATA));

		// set default monospaced font
		$pdf->SetDefaultMonospacedFont(PDF_FONT_MONOSPACED);

		//set margins
		$pdf->SetMargins(PDF_MARGIN_LEFT, PDF_MARGIN_TOP, PDF_MARGIN_RIGHT);
		$pdf->SetHeaderMargin(PDF_MARGIN_HEADER);
		$pdf->SetFooterMargin(PDF_MARGIN_FOOTER);

		//set auto page breaks
		$pdf->SetAutoPageBreak(TRUE, PDF_MARGIN_BOTTOM);

		//set image scale factor
		$pdf->setImageScale(PDF_IMAGE_SCALE_RATIO);

		//set some language-dependent strings
		$pdf->setLanguageArray($l);

		// set font
		$pdf->SetFont('helvetica', 'B', 16);
		
		// add a page
		$pdf->AddPage();
		$title = $table->title;
		$pdf->Write(0, $title, '', 0, 'L', true, 0, false, false, 0);
		$pdf->SetFont('dejavusans', '', 8);

   		$html = $this->buildhtmltable($table);

   		$pdf->writeHTML($html, true, false, false, false, '');
   		$pdf->Output('export.pdf', 'D');
   	}
   	
   	public function event_availability($startdate, $enddate, $maxUsers, $users, $classroom_id) {
	
		global $wpdb;
		$table = $wpdb->prefix . 'tlive_rooms';
	    /* Query to search all events that intersect with current */
	    $strsqlstring = "SELECT * FROM $table AS c WHERE id <> %d AND (timestarted < %s AND timecompleted > %s) ORDER BY timestarted, timecompleted ASC";
	    
		/* Sliding window of aux array for comparison */
	    $count = 0;    
	    
	    if ($res = $wpdb->get_results( $wpdb->prepare($strsqlstring, $classroom_id, $enddate, $startdate) )) {
	        /* Nested loop that make comparisons between time interval, to count total participants */			
				
	        foreach ($res as $i) {
	            /* Subarray used to make comparisons between events */
	            $subarray = array_slice($res, $count+1, count($res));
	            $usersCount = 0;
				
	            foreach ($subarray as $j) {
		
	            	$start = new DateTime($j->timestarted, new DatetimeZone('UTC'));	            	
	            	$end   = new DateTime($i->timecompleted, new DatetimeZone('UTC'));
	            	
	                /* Checks if two time intervals intersect, then makes users partial sum */
	                if ($end > $start) {
	                    /* somma parziale */
	                    $usersCount = $usersCount + $j->users;
	                }
	            }
	
	            /* Total sum */
	            $usersCount = $usersCount + $users + $i->users;
	
	            if ($usersCount > $maxUsers) {
	                /* If total users is greater than max users, return false  */           
	                return false;
	            }
	            $count++;
	        }
	        /* if there aren't any events that intersect the current */
	    } else {
	        if ($users > $maxUsers) {
	            /* If total users is greater than max users, return false  */           
	            return false;
	        }
	    }
	    return true;
	}
}
